<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Carbon\Carbon;

class SeasonalType extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'description',
        'default_start_date',
        'default_end_date',
        'is_active',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'default_start_date' => 'date',
        'default_end_date' => 'date',
        'is_active' => 'boolean',
    ];

    /**
     * Get the coupons associated with this seasonal type.
     */
    public function coupons(): HasMany
    {
        return $this->hasMany(Coupon::class);
    }

    /**
     * Scope a query to only include active seasonal types.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Check if the current date falls within this seasonal period.
     *
     * @return bool
     */
    public function isCurrentlyActive(): bool
    {
        if (!$this->is_active || !$this->default_start_date || !$this->default_end_date) {
            return false;
        }

        $now = Carbon::now();
        $year = $now->year;

        // Create dates for the current year
        $start = Carbon::createFromFormat('Y-m-d', $year . '-' . $this->default_start_date->format('m-d'));
        $end = Carbon::createFromFormat('Y-m-d', $year . '-' . $this->default_end_date->format('m-d'));

        // If end date is before start date, it spans across years (e.g., winter break Dec-Jan)
        if ($end->lt($start)) {
            $end->addYear();

            // If current date is in the later part of the period (January)
            if ($now->month < 6) {
                $start->subYear();
            }
        }

        return $now->between($start, $end);
    }

    /**
     * Get currently active seasonal types.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getCurrentlyActive()
    {
        // This implementation uses filter() which returns a support collection
        // Let's modify it to return an Eloquent collection

        $activeSeasons = self::active()
            ->get()
            ->filter(function ($seasonalType) {
                return $seasonalType->isCurrentlyActive();
            });

        // Convert back to an Eloquent Collection
        return self::whereIn('id', $activeSeasons->pluck('id'))->get();
    }
}
