<?php

namespace App\Models;

use App\Models\Order;
use Illuminate\Notifications\Notifiable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable implements MustVerifyEmail
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'user_type',
        'status',
        'country_id',
        'phone',
        'billing_street_address',
        'billing_city',
        'billing_state_province',
        'billing_postal_code',
        'billing_country_id',
        'tax_id',
        'company_name',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    /**
     * Get messages sent by this user.
     */
    public function messages()
    {
        return $this->hasMany(\App\Models\Message::class, 'sender_id');
    }

    /**
     * Get conversations this user participates in.
     */
    public function participatingConversations()
    {
        return $this->belongsToMany(\App\Models\Conversation::class, 'conversation_participants');
    }

    /**
     * Get the country associated with the user.
     */
    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    // Relationship with WriterProfile model
    public function writerProfile()
    {
        return $this->hasOne(WriterProfile::class);
    }

    /**
     * Get the client preferences for the user.
     */
    public function clientPreferences()
    {
        return $this->hasOne(ClientPreference::class);
    }

    /**
     * Get the billing country.
     */
    public function billingCountry()
    {
        return $this->belongsTo(Country::class, 'billing_country_id');
    }

    /**
     * Check if the user is a writer.
     */
    public function isWriter()
    {
        return $this->user_type === 'writer';
    }

    /**
     * Check if the user is a super admin.
     */
    public function isSuperAdmin()
    {
        return $this->user_type === 'super_admin';
    }


    /**
     * Check if the user is an admin.
     */
    public function isAdmin()
    {
        return $this->user_type === 'admin';
    }

    /**
     * Check if the user is a client.
     */
    public function isClient()
    {
        return $this->user_type === 'client';
    }

    /**
     * Get the writer's test responses.
     */
    public function testResponses()
    {
        return $this->hasMany(WriterTestResponse::class);
    }

    /**
     * Get the writer's bids.
     */
    public function bids()
    {
        return $this->hasMany(Bid::class, 'writer_id');
    }

    /**
     * Get ratings given by this user (as a client).
     */
    public function ratingsGiven()
    {
        return $this->hasMany(OrderRating::class, 'client_id');
    }

    /**
     * Get ratings received by this user (as a writer).
     */
    public function ratingsReceived()
    {
        return $this->hasMany(OrderRating::class, 'writer_id');
    }

    /**
     * Get the average rating received as a writer.
     */
    public function getAverageRating(): float
    {
        $rating = $this->ratingsReceived()
            ->verified()
            ->avg('overall_rating');
        
        return round($rating ?? 0, 2);
    }

    /**
     * Get the total number of ratings received as a writer.
     */
    public function getTotalRatingsCount(): int
    {
        return $this->ratingsReceived()
            ->verified()
            ->count();
    }

    /**
     * Get the rating distribution (count of each star rating).
     */
    public function getRatingDistribution(): array
    {
        $distribution = [];
        for ($i = 1; $i <= 5; $i++) {
            $distribution[$i] = $this->ratingsReceived()
                ->verified()
                ->ratingRange($i)
                ->count();
        }
        return $distribution;
    }

    /**
     * Get the percentage of positive ratings (4-5 stars).
     */
    public function getPositiveRatingPercentage(): float
    {
        $totalRatings = $this->getTotalRatingsCount();
        if ($totalRatings === 0) {
            return 0;
        }

        $positiveRatings = $this->ratingsReceived()
            ->verified()
            ->where('overall_rating', '>=', 4)
            ->count();

        return round(($positiveRatings / $totalRatings) * 100, 1);
    }

    /**
     * Check if user has enough ratings to be considered rated.
     */
    public function hasEnoughRatings(int $minimum = 3): bool
    {
        return $this->getTotalRatingsCount() >= $minimum;
    }

    /**
     * Get the count of active orders for the writer.
     *
     * @return int
     */
    // public function activeOrdersCount()
    // {
    //     return $this->orders()->where('order_status', '!=', 'assigned')->count();
    // }

	public function activeOrdersCount()
{
    return $this->orders()
                ->where('order_status', 'assigned')
                ->count();
}


    /**
     * Get the count of approved orders for the writer.
     *
     * @return int
     */
    public function approvedOrdersCount()
    {
        return $this->orders()
            ->where('order_status', 'approved') // Use 'approved' status
            ->where('payment_status', 'paid')
            ->count();
    }

    /**
     * Get the list of previous orders completed by the writer for a specific client.
     *
     * @param int $clientId
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function previousOrdersForClient($clientId)
    {
        return $this->orders()
            ->where('user_id', $clientId)
            ->where('order_status', 'approved') // Use 'approved' status
            ->get();
    }

    /**
     * Get notification preferences for the user.
     */
    public function notificationPreferences()
    {
        return $this->hasOne(NotificationPreference::class);
    }

    /**
     * Get the channels the user prefers for a specific notification type.
     *
     * @param string $notificationType
     * @param array $defaultChannels
     * @return array
     */
    public function preferredChannels(string $notificationType, array $defaultChannels = ['database', 'mail']): array
    {
        $preference = $this->notificationPreferences()
            ->where('notification_type', $notificationType)
            ->first();

        if (!$preference) {
            return $defaultChannels;
        }

        $channels = [];

        // Always include database channel for in-app notifications
        if ($preference->in_app_enabled) {
            $channels[] = 'database';
        }

        // Add email channel if enabled
        if ($preference->email_enabled) {
            $channels[] = 'mail';
        }

        // Add other channels as they become supported
        if ($preference->push_enabled) {
            $channels[] = 'broadcast';
        }

        // If no channels are enabled, default to database at minimum
        return empty($channels) ? ['database'] : $channels;
    }

    /**
     * Check if user has enabled email for a specific notification type.
     *
     * @param string $notificationType
     * @param bool $default
     * @return bool
     */
    public function hasEmailPreference(string $notificationType, bool $default = true): bool
    {
        $preference = $this->notificationPreferences()
            ->where('notification_type', $notificationType)
            ->first();

        if (!$preference) {
            return $default;
        }

        return $preference->email_enabled;
    }

    // Writer-specific relationships for the new systems

    /**
     * Get the revisions assigned to this writer.
     */
    public function assignedRevisions()
    {
        return $this->hasMany(OrderRevision::class, 'assigned_to');
    }

    /**
     * Get the active revisions assigned to this writer.
     */
    public function activeRevisions()
    {
        return $this->hasMany(OrderRevision::class, 'assigned_to')
                    ->whereIn('status', ['requested', 'in_progress']);
    }

    /**
     * Get the orders where this writer was the original assignee.
     */
    public function originalOrderAssignments()
    {
        return $this->hasMany(OrderReassignment::class, 'original_writer_id');
    }

    /**
     * Get the orders where this writer was reassigned to.
     */
    public function newOrderAssignments()
    {
        return $this->hasMany(OrderReassignment::class, 'new_writer_id');
    }

    /**
     * Get all reassignments involving this writer.
     */
    public function orderReassignments()
    {
        return $this->hasMany(OrderReassignment::class, 'original_writer_id')
                    ->orWhere('new_writer_id', $this->id);
    }

    /**
     * Get the infractions for this writer.
     */
    public function infractions()
    {
        return $this->hasMany(WriterInfraction::class, 'writer_id');
    }

    /**
     * Get the active infractions for this writer.
     */
    public function activeInfractions()
    {
        return $this->hasMany(WriterInfraction::class, 'writer_id')
                    ->whereIn('status', ['reported', 'under_review', 'confirmed']);
    }

    /**
     * Get the fines for this writer.
     */
    public function fines()
    {
        return $this->hasMany(WriterFine::class, 'writer_id');
    }

    /**
     * Get the active fines for this writer.
     */
    public function activeFines()
    {
        return $this->hasMany(WriterFine::class, 'writer_id')
                    ->whereIn('status', ['pending', 'applied']);
    }

    /**
     * Get the total amount of active fines for this writer.
     */
    public function getTotalActiveFines(): float
    {
        return $this->activeFines()->sum('remaining_amount');
    }

    /**
     * Check if this writer has any active infractions.
     */
    public function hasActiveInfractions(): bool
    {
        return $this->activeInfractions()->exists();
    }

    /**
     * Check if this writer has any active fines.
     */
    public function hasActiveFines(): bool
    {
        return $this->activeFines()->exists();
    }

    /**
     * Get the writer's performance metrics based on infractions and reassignments.
     */
    public function getPerformanceMetrics(): array
    {
        $totalOrders = $this->orders()->where('order_status', 'approved')->count();
        $reassignments = $this->originalOrderAssignments()->count();
        $infractions = $this->infractions()->count();
        $activeInfractions = $this->activeInfractions()->count();
        $totalFines = $this->fines()->sum('fine_amount');

        return [
            'total_orders' => $totalOrders,
            'reassignments' => $reassignments,
            'reassignment_rate' => $totalOrders > 0 ? ($reassignments / $totalOrders) * 100 : 0,
            'infractions' => $infractions,
            'active_infractions' => $activeInfractions,
            'total_fines' => $totalFines,
            'performance_score' => $this->calculatePerformanceScore($totalOrders, $reassignments, $infractions, $totalFines),
        ];
    }

    /**
     * Calculate a performance score based on various metrics.
     */
    private function calculatePerformanceScore(int $totalOrders, int $reassignments, int $infractions, float $totalFines): float
    {
        if ($totalOrders === 0) {
            return 0;
        }

        $baseScore = 100;
        
        // Deduct points for reassignments (5 points each)
        $baseScore -= ($reassignments * 5);
        
        // Deduct points for infractions (10 points each)
        $baseScore -= ($infractions * 10);
        
        // Deduct points for fines (1 point per dollar)
        $baseScore -= ($totalFines * 1);
        
        return max(0, $baseScore);
    }
}
