<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class WriterPaymentRecord extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'writer_id',
        'payment_amount',
        'currency',
        'status',
        'earned_at',
        'available_at',
        'withdrawn_at',
        'withdrawal_request_id',
        'withdrawal_batch_id',
        'payment_method',
        'transaction_reference',
        'notes',
        'metadata',
    ];

    protected $casts = [
        'payment_amount' => 'decimal:2',
        'earned_at' => 'datetime',
        'available_at' => 'datetime',
        'withdrawn_at' => 'datetime',
        'metadata' => 'array',
    ];

    // Payment status constants
    const STATUS_PENDING = 'pending';
    const STATUS_AVAILABLE = 'available';
    const STATUS_WITHDRAWN = 'withdrawn';
    const STATUS_PROCESSING = 'processing';
    const STATUS_DISPUTED = 'disputed';

    /**
     * Get the order that this payment record belongs to.
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the writer that this payment record belongs to.
     */
    public function writer()
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Get the withdrawal request associated with this payment.
     */
    public function withdrawalRequest()
    {
        return $this->belongsTo(WriterWithdrawalRequest::class, 'withdrawal_request_id');
    }

    /**
     * Scope to get pending payments for a writer.
     */
    public function scopePending($query, $writerId = null)
    {
        $query = $query->where('status', self::STATUS_PENDING);
        if ($writerId) {
            $query->where('writer_id', $writerId);
        }
        return $query;
    }

    /**
     * Scope to get available payments for a writer.
     */
    public function scopeAvailable($query, $writerId = null)
    {
        $query = $query->where('status', self::STATUS_AVAILABLE);
        if ($writerId) {
            $query->where('writer_id', $writerId);
        }
        return $query;
    }

    /**
     * Scope to get withdrawn payments for a writer.
     */
    public function scopeWithdrawn($query, $writerId = null)
    {
        $query = $query->where('status', self::STATUS_WITHDRAWN);
        if ($writerId) {
            $query->where('writer_id', $writerId);
        }
        return $query;
    }

    /**
     * Mark payment as available for withdrawal.
     */
    public function markAsAvailable()
    {
        $this->update([
            'status' => self::STATUS_AVAILABLE,
            'available_at' => now(),
        ]);
    }

    /**
     * Mark payment as withdrawn.
     */
    public function markAsWithdrawn($withdrawalRequestId = null, $transactionReference = null)
    {
        $this->update([
            'status' => self::STATUS_WITHDRAWN,
            'withdrawn_at' => now(),
            'withdrawal_request_id' => $withdrawalRequestId,
            'transaction_reference' => $transactionReference,
        ]);
    }
}
