<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class WriterWithdrawalRequest extends Model
{
    use HasFactory;

    protected $fillable = [
        'writer_id',
        'requested_amount',
        'currency',
        'withdrawal_date',
        'status',
        'payment_method',
        'payment_details',
        'approved_by',
        'approved_at',
        'processed_at',
        'paid_at',
        'withdrawal_batch_id',
        'transaction_reference',
        'notes',
        'rejection_reason',
        'metadata',
    ];

    protected $casts = [
        'requested_amount' => 'decimal:2',
        'withdrawal_date' => 'date',
        'payment_details' => 'array',
        'approved_at' => 'datetime',
        'processed_at' => 'datetime',
        'paid_at' => 'datetime',
        'metadata' => 'array',
    ];

    // Status constants
    const STATUS_REQUESTED = 'requested';
    const STATUS_APPROVED = 'approved';
    const STATUS_PROCESSING = 'processing';
    const STATUS_PAID = 'paid';
    const STATUS_REJECTED = 'rejected';
    const STATUS_CANCELLED = 'cancelled';

    // Valid withdrawal dates
    const VALID_WITHDRAWAL_DAYS = [15, 30]; // Last day of month will be converted to 30

    /**
     * Get the writer that made this withdrawal request.
     */
    public function writer()
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Get the admin who approved this request.
     */
    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get the payment records associated with this withdrawal.
     */
    public function paymentRecords()
    {
        return $this->hasMany(WriterPaymentRecord::class, 'withdrawal_request_id');
    }

    /**
     * Scope to get pending withdrawal requests.
     */
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_REQUESTED);
    }

    /**
     * Scope to get approved withdrawal requests.
     */
    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }

    /**
     * Scope to get withdrawal requests for a specific date.
     */
    public function scopeForDate($query, $date)
    {
        return $query->whereDate('withdrawal_date', $date);
    }

    /**
     * Check if the withdrawal date is valid (15th or 30th/last day of month).
     */
    public static function isValidWithdrawalDate($date)
    {
        $day = $date->day;
        $lastDayOfMonth = $date->copy()->endOfMonth()->day;
        
        return $day === 15 || $day === $lastDayOfMonth || $day === 30;
    }

    /**
     * Get the next valid withdrawal date.
     */
    public static function getNextWithdrawalDate()
    {
        $now = now();
        $currentDay = $now->day;
        
        // If we're before the 15th, next withdrawal is 15th of current month
        if ($currentDay < 15) {
            return $now->copy()->day(15);
        }
        // If we're on or after 15th but before last day, next withdrawal is last day of current month
        elseif ($currentDay < $now->copy()->endOfMonth()->day) {
            return $now->copy()->endOfMonth();
        }
        // Otherwise, next withdrawal is 15th of next month
        else {
            return $now->copy()->addMonth()->day(15);
        }
    }

    /**
     * Approve the withdrawal request.
     */
    public function approve($approvedBy)
    {
        $this->update([
            'status' => self::STATUS_APPROVED,
            'approved_by' => $approvedBy,
            'approved_at' => now(),
        ]);
    }

    /**
     * Reject the withdrawal request.
     */
    public function reject($reason, $rejectedBy = null)
    {
        $this->update([
            'status' => self::STATUS_REJECTED,
            'rejection_reason' => $reason,
            'approved_by' => $rejectedBy, // Track who rejected it
            'approved_at' => now(),
        ]);
    }

    /**
     * Mark as processing.
     */
    public function markAsProcessing()
    {
        $this->update([
            'status' => self::STATUS_PROCESSING,
            'processed_at' => now(),
        ]);
    }

    /**
     * Mark as paid.
     */
    public function markAsPaid($transactionReference = null)
    {
        $this->update([
            'status' => self::STATUS_PAID,
            'paid_at' => now(),
            'transaction_reference' => $transactionReference,
        ]);
    }
}
