<?php

namespace App\Notifications;

use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminEssayReviewNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $writerName;
    protected $writerEmail;
    protected $essayId;
    protected $essayTopic;
    protected $submissionDate;
    protected $fileUrl;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        string $writerName,
        string $writerEmail,
        int $essayId,
        string $essayTopic,
        string $submissionDate,
        ?string $fileUrl = null
    ) {
        $this->writerName = $writerName;
        $this->writerEmail = $writerEmail;
        $this->essayId = $essayId;
        $this->essayTopic = $essayTopic;
        $this->submissionDate = $submissionDate;
        $this->fileUrl = $fileUrl;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('New Essay Submission for Review')
            ->greeting('Hello Admin,')
            ->line("A new essay has been submitted and requires your review.")
            ->line("Writer: {$this->writerName} ({$this->writerEmail})")
            ->line("Topic: {$this->essayTopic}")
            ->line("Submitted: {$this->submissionDate}")
            ->action('Review Essay', url("/admin/essays/{$this->essayId}"));

        if ($this->fileUrl) {
            $message->line("The essay file is available for download in the admin panel.");
        }

        return $message->line('Thank you for your attention to this matter.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Create view essay action
        $action = $this->createAction(
            'Review Essay',
            url("/admin/essays/{$this->essayId}"),
            'GET',
            'primary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'essay_submission',
            "New Essay Submission",
            "A new essay has been submitted by {$this->writerName} and requires your review.",
            'file-text',
            'info',
            'high',  // High priority for admin review tasks
            'essay',
            [$action],
            now()->addDays(14),  // Expires after 14 days
            [
                'writer_name' => $this->writerName,
                'writer_email' => $this->writerEmail,
                'essay_id' => $this->essayId,
                'essay_topic' => $this->essayTopic,
                'submission_date' => $this->submissionDate,
                'has_file' => !empty($this->fileUrl),
            ]
        );
    }
}
