<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\Order;
use App\Models\OrderFile;

class AdminOrderFileUploadedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $order;
    public $orderFile;

    /**
     * Create a new notification instance.
     */
    public function __construct(Order $order, OrderFile $orderFile)
    {
        $this->order = $order;
        $this->orderFile = $orderFile;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $fileCategory = ucfirst($this->orderFile->file_category);
        $clientName = $this->order->user->name;
        $orderNumber = $this->order->order_number;

        return (new MailMessage)
            ->subject("New {$fileCategory} File Uploaded - Order #{$orderNumber}")
            ->greeting("Hello {$notifiable->name},")
            ->line("A client has uploaded a new {$fileCategory} file for order #{$orderNumber}.")
            ->line("Client: {$clientName}")
            ->line("File: {$this->orderFile->file_name}")
            ->line("Category: {$fileCategory}")
            ->line("Description: " . ($this->orderFile->description ?: 'No description provided'))
            ->action('View Order', url("/admin/orders/{$this->order->id}"))
            ->line('Please review the uploaded file and take any necessary actions.')
            ->salutation('Best regards, AcademicScribe System');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'order_file_uploaded',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'file_id' => $this->orderFile->id,
            'file_name' => $this->orderFile->file_name,
            'file_category' => $this->orderFile->file_category,
            'client_name' => $this->order->user->name,
            'uploaded_at' => $this->orderFile->date_uploaded,
            'message' => "Client {$this->order->user->name} uploaded a {$this->orderFile->file_category} file for order #{$this->order->order_number}",
            'action_url' => "/admin/orders/{$this->order->id}",
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'order_file_uploaded',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'file_id' => $this->orderFile->id,
            'file_name' => $this->orderFile->file_name,
            'file_category' => $this->orderFile->file_category,
            'client_name' => $this->order->user->name,
            'uploaded_at' => $this->orderFile->date_uploaded,
            'message' => "Client {$this->order->user->name} uploaded a {$this->orderFile->file_category} file for order #{$this->order->order_number}",
            'action_url' => "/admin/orders/{$this->order->id}",
        ];
    }
} 