<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\Payment;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminPaymentNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $payment;
    protected $userName;
    protected $isFailure;
    protected $errorMessage;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param Payment $payment
     * @param string $userName
     * @param bool $isFailure
     * @param string|null $errorMessage
     */
    public function __construct(
        Order $order,
        Payment $payment,
        string $userName,
        bool $isFailure = false,
        ?string $errorMessage = null
    ) {
        $this->order = $order;
        $this->payment = $payment;
        $this->userName = $userName;
        $this->isFailure = $isFailure;
        $this->errorMessage = $errorMessage;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Format amount for display
        $amount = '$' . number_format($this->payment->amount, 2);

        // Format payment date
        $paymentDate = $this->payment->payment_date
            ? $this->payment->payment_date->format('F j, Y, g:i a')
            : now()->format('F j, Y, g:i a');

        // Format payment method for display
        $paymentMethod = ucfirst(str_replace('_', ' ', $this->payment->payment_method));

        // Get admin name if available
        $adminName = $notifiable->name ?? 'Admin';

        // Create appropriate subject and content based on success/failure
        $subject = $this->isFailure
            ? "Payment Failed - Order #{$this->order->order_number}"
            : "Payment Received - Order #{$this->order->order_number}";

        $mailMessage = (new MailMessage)
            ->subject($subject)
            ->greeting("Hello {$adminName},");

        if ($this->isFailure) {
            $mailMessage->line("A payment for Order #{$this->order->order_number} has failed.")
                ->line("Customer: {$this->userName}")
                ->line("Error: " . ($this->errorMessage ?? 'Unknown error'));
        } else {
            $mailMessage->line("A payment for Order #{$this->order->order_number} has been received.")
                ->line("Customer: {$this->userName}");
        }

        $mailMessage->line('Payment Details:')
            ->line('Amount: ' . $amount)
            ->line('Payment Method: ' . $paymentMethod)
            ->line('Transaction ID: ' . ($this->payment->transaction_id ?? 'N/A'))
            ->line('Payment Date: ' . $paymentDate)
            ->line('Status: ' . ucfirst($this->payment->status))
            ->action('View Order Details', url("/admin/orders/{$this->order->id}"));

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Format amount for display
        $amount = '$' . number_format($this->payment->amount, 2);

        // Create view order action
        $viewOrderAction = $this->createAction(
            'View Order',
            url("/admin/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        // Create view payments action
        $viewPaymentsAction = $this->createAction(
            'All Payments',
            url("/admin/payments"),
            'GET',
            'secondary'
        );

        // Determine notification details based on success or failure
        $type = $this->isFailure ? 'payment_failed_admin' : 'payment_received_admin';
        $title = $this->isFailure ? 'Payment Failed' : 'Payment Received';
        $message = $this->isFailure
            ? "Payment of {$amount} for Order #{$this->order->order_number} by {$this->userName} has failed."
            : "Payment of {$amount} for Order #{$this->order->order_number} by {$this->userName} has been received.";
        $icon = $this->isFailure ? 'alert-circle' : 'credit-card';
        $color = $this->isFailure ? 'danger' : 'success';

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            $type,
            $title,
            $message,
            $icon,
            $color,
            $this->isFailure ? 'high' : 'medium', // Higher priority for failures
            'payment',
            [$viewOrderAction, $viewPaymentsAction],
            now()->addDays($this->isFailure ? 7 : 30), // Expires sooner for failures
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'payment_id' => $this->payment->id,
                'payment_method' => $this->payment->payment_method,
                'amount' => $this->payment->amount,
                'transaction_id' => $this->payment->transaction_id,
                'payment_date' => $this->payment->payment_date ? $this->payment->payment_date->toISOString() : now()->toISOString(),
                'status' => $this->payment->status,
                'customer_name' => $this->userName,
                'customer_id' => $this->order->user_id,
                'is_failure' => $this->isFailure,
                'error_message' => $this->errorMessage,
            ]
        );
    }
}
