<?php

namespace App\Notifications;

use App\Models\User;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;

class AdminUserRegisterNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $user;
    protected $password;

    /**
     * Create a new notification instance.
     *
     * @param User $user
     * @param string|null $password
     */
    public function __construct(User $user, ?string $password = null)
    {
        $this->user = $user;
        $this->password = $password;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('Welcome to ' . config('app.name'))
            ->greeting('Hello ' . $this->user->name . '!')
            ->line('An account has been created for you on ' . config('app.name') . '.')
            ->line('Your login email is: ' . $this->user->email);

        if ($this->password) {
            $message->line('Your temporary password is: ' . $this->password)
                ->line('Please change your password after logging in for the first time.');
        }

        $message->action('Login to Your Account', url('/login'))
            ->line('Thank you for using our Services!');

        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Create login action
        $action = $this->createAction(
            'Login Now',
            url('/login'),
            'GET',
            'primary'
        );

        // Create profile action
        $profileAction = $this->createAction(
            'Update Profile',
            url('/profile'),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'account_created',
            "Welcome to " . config('app.name'),
            "Your account has been created by an administrator. Please login to access your account.",
            'user-plus',
            'info',
            'medium',
            'account',
            [$action, $profileAction],
            now()->addDays(30),  // Expires after 30 days
            [
                'user_id' => $this->user->id,
                'user_name' => $this->user->name,
                'user_email' => $this->user->email,
                'user_type' => $this->user->user_type,
                'has_temp_password' => !is_null($this->password),
                'created_at' => now()->toISOString(),
            ]
        );
    }
}
