<?php

namespace App\Notifications;

use App\Models\Bid;
use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class BidAcceptedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $bid;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param Bid $bid
     */
    public function __construct(Order $order, Bid $bid)
    {
        $this->order = $order;
        $this->bid = $bid;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('bid_accepted', ['mail', 'database']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Format the deadline for display
        $deadline = $this->order->deadline
            ? \Carbon\Carbon::parse($this->order->deadline)->format('F j, Y, g:i a')
            : 'As specified in order details';

        // Format the bid amount with currency symbol
        $bidAmount = '$' . number_format($this->bid->bid_amount, 2);

        return (new MailMessage)
            ->subject('Your Bid Has Been Accepted!')
            ->greeting('Congratulations!')
            ->line('Your bid for Order #' . $this->order->order_number . ' has been accepted.')
            ->line('Title: ' . $this->order->title)
            ->line('Bid Amount: ' . $bidAmount)
            ->line('Completion Time: ' . date('F j, Y, g:i a', strtotime($this->bid->completion_time)))
            ->line('Deadline: ' . $deadline)
            ->action('View Order Details', url('/writer/orders/' . $this->order->id))
            ->line('Please complete the order by the agreed deadline.')
            ->line('Thank you for being part of our writing team!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array
     */
    public function toArray(object $notifiable): array
    {
        // Create view order action
        $action = $this->createAction(
            'View Order',
            url('/writer/orders/' . $this->order->id),
            'GET',
            'primary'
        );

        // Format the deadline for display
        $deadline = $this->order->deadline
            ? \Carbon\Carbon::parse($this->order->deadline)->format('F j, Y, g:i a')
            : 'As specified in order details';

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'bid_accepted',
            "Bid Accepted for Order #{$this->order->order_number}",
            "Congratulations! Your bid for Order #{$this->order->order_number} ({$this->order->title}) has been accepted.",
            'check-circle',
            'success',
            'high',
            'order',
            [$action],
            now()->addDays(7), // Expires after 7 days
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'bid_amount' => $this->bid->bid_amount,
                'formatted_bid_amount' => '$' . number_format($this->bid->bid_amount, 2),
                'completion_time' => $this->bid->completion_time,
                'formatted_completion_time' => date('F j, Y, g:i a', strtotime($this->bid->completion_time)),
                'deadline' => $this->order->deadline,
                'formatted_deadline' => $deadline,
                'pages' => $this->order->pages,
                'subject' => $this->order->subject,
                'accepted_at' => now()->toISOString(),
            ]
        );
    }
}
