<?php

namespace App\Notifications;

use App\Models\User;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ClientRegistered extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $user;

    /**
     * Create a new notification instance.
     *
     * @param User $user
     */
    public function __construct(User $user)
    {
        $this->user = $user;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Welcome to ' . config('app.name') . '!')
            ->greeting("Hello {$this->user->name},")
            ->line('Thank you for registering with our platform. We are excited to have you on board!')
            ->line('A confirmation email has been sent to your email address. You will need to confirm your email before you can fully access your account.')
            ->action('Login to Your Account', url('/login'))
            ->line('If you need any assistance, our support team is here to help.')
            ->line('Thank you for choosing the best Essay Writing Service!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Create login action
        $loginAction = $this->createAction(
            'Login Now',
            url('/login'),
            'GET',
            'primary'
        );

        // Create contact support action
        $supportAction = $this->createAction(
            'Contact Support',
            url('/support'),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'client_registered',
            "Welcome to " . config('app.name'),
            "Thank you for registering, {$this->user->name}! Please verify your email to start using our services.",
            'user-plus',
            'success',
            'high',
            'account',
            [$loginAction, $supportAction],
            now()->addDays(30), // Expires after 30 days
            [
                'user_id' => $this->user->id,
                'user_name' => $this->user->name,
                'user_email' => $this->user->email,
                'registered_at' => now()->toISOString(),
                'needs_verification' => true,
            ]
        );
    }
}
