<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\OrderFile;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;

class ClientWorkCompletedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $order;
    protected $orderFile;

    /**
     * Create a new notification instance.
     */
    public function __construct(Order $order, OrderFile $orderFile)
    {
        $this->order = $order;
        $this->orderFile = $orderFile;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $orderNumber = $this->order->order_number;
        $title = $this->order->title;
        $fileName = $this->orderFile->file_name;

        return (new MailMessage)
            ->subject("Work Completed - Order #{$orderNumber}")
            ->greeting("Hello {$notifiable->name}!")
            ->line("Great news! Your writer has completed the work for your order.")
            ->line("**Order Details:**")
            ->line("• Order Number: #{$orderNumber}")
            ->line("• Title: {$title}")
            ->line("• File: {$fileName}")
            ->line("Your order is now under admin review. Once approved, you'll be able to download the completed work.")
            ->action('View Order Details', route('orders.show', $this->order->id))
            ->line('Thank you for choosing our service!')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'work_completed',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'title' => $this->order->title,
            'file_name' => $this->orderFile->file_name,
            'message' => "Your writer has completed the work for order #{$this->order->order_number}. The order is now under admin review.",
            'timestamp' => now()->toISOString(),
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'work_completed',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'title' => $this->order->title,
            'file_name' => $this->orderFile->file_name,
            'message' => "Your writer has completed the work for order #{$this->order->order_number}. The order is now under admin review.",
            'timestamp' => now()->toISOString(),
        ];
    }
} 