<?php

namespace App\Notifications;

use App\Models\ContactMessage;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ContactMessageAdminNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $contactMessage;

    /**
     * Create a new notification instance.
     */
    public function __construct(ContactMessage $contactMessage)
    {
        $this->contactMessage = $contactMessage;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('New Contact Message - ' . $this->contactMessage->subject)
            ->greeting('Hello Admin,')
            ->line('A new contact message has been submitted on the website.')
            ->line('**Contact Details:**')
            ->line('**Name:** ' . $this->contactMessage->name)
            ->line('**Email:** ' . $this->contactMessage->email)
            ->line('**Phone:** ' . ($this->contactMessage->phone ?: 'Not provided'))
            ->line('**Subject:** ' . $this->contactMessage->subject)
            ->line('**Message:**')
            ->line($this->contactMessage->message)
            ->line('**Submitted on:** ' . $this->contactMessage->created_at->format('F j, Y \a\t g:i A'))
            ->line('**IP Address:** ' . $this->contactMessage->ip_address)
            ->action('View in Admin Panel', url('/admin/contact-messages/' . $this->contactMessage->id))
            ->line('Please respond to this message promptly to maintain good customer service.');
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray($notifiable): array
    {
        // Create view message action
        $viewAction = $this->createAction(
            'View Message',
            url('/admin/contact-messages/' . $this->contactMessage->id),
            'GET',
            'primary'
        );

        // Create reply action
        $replyAction = $this->createAction(
            'Reply',
            url('/admin/contact-messages/' . $this->contactMessage->id . '/reply'),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'contact_message',
            'New Contact Message',
            "New contact message from {$this->contactMessage->name} - {$this->contactMessage->subject}",
            'mail',
            'info',
            'high',
            'contact',
            [$viewAction, $replyAction],
            now()->addDays(30), // Expires after 30 days
            [
                'contact_message_id' => $this->contactMessage->id,
                'sender_name' => $this->contactMessage->name,
                'sender_email' => $this->contactMessage->email,
                'subject' => $this->contactMessage->subject,
                'message_preview' => substr($this->contactMessage->message, 0, 100) . '...',
                'phone' => $this->contactMessage->phone,
                'submitted_at' => $this->contactMessage->created_at->toISOString(),
                'ip_address' => $this->contactMessage->ip_address,
                'status' => $this->contactMessage->status,
            ]
        );
    }
}
