<?php

namespace App\Notifications;

use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class EssayReviewNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $isApproved;
    protected $feedback;
    protected $essayId;
    protected $essayTopic;
    protected $reviewerName;

    /**
     * Create a new notification instance.
     *
     * @param bool $isApproved
     * @param string|null $feedback
     * @param int|null $essayId
     * @param string|null $essayTopic
     * @param string|null $reviewerName
     */
    public function __construct(
        bool $isApproved,
        ?string $feedback,
        ?int $essayId = null,
        ?string $essayTopic = null,
        ?string $reviewerName = 'Review Team'
    ) {
        $this->isApproved = $isApproved;
        $this->feedback = $feedback;
        $this->essayId = $essayId;
        $this->essayTopic = $essayTopic;
        $this->reviewerName = $reviewerName;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $mailMessage = (new MailMessage)
            ->subject($this->isApproved ? 'Congratulations! Your Writer Application is Approved' : 'Writer Application Update: Essay Review Results')
            ->greeting('Hello ' . $notifiable->name . ',');

        if ($this->isApproved) {
            $mailMessage
                ->line('Great news! Your essay submission has been approved.')
                ->line('You have successfully completed the writer qualification process and can now start bidding on available orders.')
                ->line('Your account has been fully activated as a writer on our platform.')
                ->action('Browse Available Orders', url(route('writer.orders')));

            if ($this->feedback) {
                $mailMessage->line('Feedback from our review team:')
                    ->line($this->feedback);
            }

            $mailMessage->line('Thank you for joining our writing team!');
        } else {
            $mailMessage
                ->line('We have reviewed your essay submission and unfortunately, it did not meet our current requirements.')
                ->line('You will be able to retry the essay test after a 3-month waiting period.')
                ->line('Feedback from our review team:')
                ->line($this->feedback)
                ->line('We encourage you to use this time to improve your writing skills based on the feedback provided.')
                ->action('View Your Dashboard', url(route('writer.dashboard')))
                ->line('Thank you for your interest in our platform.');
        }

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Define action based on approval status
        $action = $this->isApproved
            ? $this->createAction(
                'Browse Orders',
                url(route('writer.orders')),
                'GET',
                'primary'
            )
            : $this->createAction(
                'View Dashboard',
                url(route('writer.dashboard')),
                'GET',
                'primary'
            );

        // Add secondary action
        $secondaryAction = $this->createAction(
            'Writer Resources',
            url('/writer/resources/guides'),
            'GET',
            'secondary'
        );

        // Define notification properties based on approval status
        $title = $this->isApproved
            ? "Essay Approved: Welcome to the Writing Team!"
            : "Essay Review Results: Not Approved";

        $message = $this->isApproved
            ? "Congratulations! Your essay has been approved. You can now bid on available orders."
            : "Your essay submission was not approved. Please check the feedback provided.";

        $icon = $this->isApproved ? "check-circle" : "x-circle";

        $color = $this->isApproved ? "success" : "warning";

        $category = "account";

        $priority = $this->isApproved ? "high" : "medium";

        // Create additional data payload
        $data = [
            'is_approved' => $this->isApproved,
            'feedback' => $this->feedback,
            'essay_id' => $this->essayId,
            'essay_topic' => $this->essayTopic,
            'reviewer_name' => $this->reviewerName,
            'reviewed_at' => now()->toISOString(),
            'retry_available_after' => $this->isApproved ? null : now()->addMonths(3)->toISOString(),
        ];

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'essay_review',
            $title,
            $message,
            $icon,
            $color,
            $priority,
            $category,
            [$action, $secondaryAction],
            now()->addMonths($this->isApproved ? 6 : 3), // Expires after 6 months if approved, 3 if not
            $data
        );
    }
}
