<?php

namespace App\Notifications;

use App\Models\WriterEssayTest;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class EssaySubmissionConfirmationNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $essayTest;
    protected $essayTopic;
    protected $isAutoSubmitted;

    /**
     * Create a new notification instance.
     *
     * @param WriterEssayTest $essayTest
     * @param string $essayTopic
     * @param bool $isAutoSubmitted
     */
    public function __construct(WriterEssayTest $essayTest, string $essayTopic, bool $isAutoSubmitted = false)
    {
        $this->essayTest = $essayTest;
        $this->essayTopic = $essayTopic;
        $this->isAutoSubmitted = $isAutoSubmitted;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $mailMessage = (new MailMessage)
            ->subject($this->isAutoSubmitted
                ? 'Essay Auto-Submitted as Time Expired'
                : 'Essay Submission Confirmation')
            ->greeting('Hello ' . $notifiable->name . ',');

        if ($this->isAutoSubmitted) {
            $mailMessage->line('Your essay on "' . $this->essayTopic . '" was automatically submitted as the time limit expired.')
                ->line('Filename: ' . $this->essayTest->original_filename)
                ->line('File size: ' . $this->formatFileSize($this->essayTest->file_size))
                ->line('Submission time: ' . now()->format('F j, Y, g:i a'))
                ->line('Your essay will be reviewed by our team, and you will be notified of the results once the review is complete.');
        } else {
            $mailMessage->line('Thank you for submitting your essay on "' . $this->essayTopic . '".')
                ->line('Your submission has been received and will be reviewed by our team.')
                ->line('Filename: ' . $this->essayTest->original_filename)
                ->line('File size: ' . $this->formatFileSize($this->essayTest->file_size))
                ->line('Submission time: ' . now()->format('F j, Y, g:i a'))
                ->line('You will be notified once the review is complete.');
        }

        return $mailMessage
            ->action('View Submission Confirmation', url(route('writer.essay.confirmation')))
            ->line('Thank you for completing this step of the writer qualification process!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Create view confirmation action
        $confirmationAction = $this->createAction(
            'View Details',
            url(route('writer.essay.confirmation')),
            'GET',
            'primary'
        );

        // Create dashboard action
        $dashboardAction = $this->createAction(
            'Return to Dashboard',
            url(route('writer.dashboard')),
            'GET',
            'secondary'
        );

        // Determine title and message based on submission type
        $title = $this->isAutoSubmitted
            ? 'Essay Auto-Submitted as Time Expired'
            : 'Essay Submission Successful';

        $message = $this->isAutoSubmitted
            ? 'Your essay on "' . $this->essayTopic . '" was automatically submitted as the time limit expired.'
            : 'Your essay on "' . $this->essayTopic . '" has been successfully submitted.';

        // Determine icon and color based on submission type
        $icon = $this->isAutoSubmitted ? 'clock' : 'check-circle';
        $color = $this->isAutoSubmitted ? 'warning' : 'success';

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'essay_submission',
            $title,
            $message,
            $icon,
            $color,
            'medium',
            'qualification',
            [$confirmationAction, $dashboardAction],
            now()->addDays(30), // Expires after 30 days
            [
                'essay_test_id' => $this->essayTest->id,
                'essay_topic' => $this->essayTopic,
                'file_name' => $this->essayTest->original_filename,
                'file_size' => $this->essayTest->file_size,
                'is_auto_submitted' => $this->isAutoSubmitted,
                'submission_time' => now()->toISOString(),
                'status' => 'pending_review',
            ]
        );
    }

    /**
     * Format file size to human-readable format.
     *
     * @param int $bytes
     * @return string
     */
    private function formatFileSize($bytes)
    {
        if ($bytes < 1024) {
            return $bytes . ' bytes';
        } elseif ($bytes < 1048576) {
            return round($bytes / 1024, 1) . ' KB';
        } else {
            return round($bytes / 1048576, 1) . ' MB';
        }
    }
}
