<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class OrderApprovedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('order_approved', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $userName = $notifiable->name ?? 'Writer';
        $clientName = $this->order->user ? $this->order->user->name : 'Your client';

        return (new MailMessage)
            ->subject("Order #{$this->order->order_number} Approved!")
            ->greeting("Congratulations {$userName}!")
            ->line("Great news! Your client has approved your work for order #{$this->order->order_number} ({$this->order->title}).")
            ->line("Client: {$clientName}")
            ->line("This order is now marked as completed and you will receive your payment according to our payment schedule.")
            ->line("Thank you for your excellent work!")
            ->action('View Order Details', url("/writer/orders/{$this->order->id}"))
            ->line('Keep up the great work!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $clientName = $this->order->user ? $this->order->user->name : 'Your client';

        // Create view order action
        $action = $this->createAction(
            'View Order',
            url("/writer/orders/{$this->order->id}"),
            'GET',
            'success'
        );

        return $this->getDatabaseNotificationData(
            'order_approved',
            "Order Approved!",
            "Your client has approved your work for order #{$this->order->order_number}. Congratulations!",
            'check-circle',
            'success',
            'high',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'client_name' => $clientName,
                'approved_at' => now()->toISOString(),
                'payment_status' => $this->order->payment_status
            ]
        );
    }
}
