<?php

namespace App\Notifications;

use App\Models\Bid;
use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class OrderAssignmentAdminNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    /**
     * The order that had a bid accepted.
     *
     * @var \App\Models\Order
     */
    protected $order;

    /**
     * The bid that was accepted.
     *
     * @var \App\Models\Bid
     */
    protected $bid;

    /**
     * Create a new notification instance.
     *
     * @param \App\Models\Order $order
     * @param \App\Models\Bid $bid
     * @return void
     */
    public function __construct(Order $order, Bid $bid)
    {
        $this->order = $order;
        $this->bid = $bid;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param mixed $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return $notifiable->preferredChannels('admin_order_writer_assigned', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param mixed $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        // Calculate profit/commission
        $profit = $this->order->net_amount - $this->bid->bid_amount;
        $profitPercentage = $this->order->net_amount > 0
            ? round(($profit / $this->order->net_amount) * 100, 2)
            : 0;

        // Format currency values
        $orderAmount = '$' . number_format($this->order->net_amount, 2);
        $bidAmount = '$' . number_format($this->bid->bid_amount, 2);
        $profitAmount = '$' . number_format($profit, 2);

        // Get client and writer names
        $clientName = $this->order->user ? $this->order->user->name : 'Unknown Client';
        $writerName = $this->bid->writer ? $this->bid->writer->name : 'Unknown Writer';

        return (new MailMessage)
            ->subject('Writer Assigned to Order #' . $this->order->order_number)
            ->greeting('Hello Admin,')
            ->line('A writer has been assigned to an order by the client.')
            ->line('**Order Details:**')
            ->line('- Order #: ' . $this->order->order_number)
            ->line('- Title: ' . $this->order->title)
            ->line('- Client: ' . $clientName . ' (ID: ' . $this->order->user_id . ')')
            ->line('- Writer: ' . $writerName . ' (ID: ' . $this->bid->writer_id . ')')
            ->line('**Financial Details:**')
            ->line('- Order Amount: ' . $orderAmount)
            ->line('- Writer Bid: ' . $bidAmount)
            ->line('- Profit: ' . $profitAmount . ' (' . $profitPercentage . '%)')
            ->line('- Completion By: ' . date('M j, Y, g:i A', strtotime($this->bid->completion_time)))
            ->action('View Order Details', url('/admin/orders/' . $this->order->id))
            ->line('The order status has been updated to "assigned".');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param mixed $notifiable
     * @return array
     */
    public function toDatabase($notifiable)
    {
        // Calculate profit/commission
        $profit = $this->order->net_amount - $this->bid->bid_amount;
        $profitPercentage = $this->order->net_amount > 0
            ? round(($profit / $this->order->net_amount) * 100, 2)
            : 0;

        // Get client and writer names
        $clientName = $this->order->user ? $this->order->user->name : 'Unknown Client';
        $writerName = $this->bid->writer ? $this->bid->writer->name : 'Unknown Writer';

        return $this->getDatabaseNotificationData(
            'admin_order_writer_assigned',
            'Writer Assigned to Order #' . $this->order->order_number,
            "$clientName assigned $writerName to their order with a profit of $" . number_format($profit, 2) . " ({$profitPercentage}%).",
            'briefcase',
            'success',
            'high',
            'order',
            [
                $this->createAction(
                    'View Order',
                    url('/admin/orders/' . $this->order->id),
                    'GET',
                    'primary'
                )
            ],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'client_id' => $this->order->user_id,
                'client_name' => $clientName,
                'bid_id' => $this->bid->id,
                'writer_id' => $this->bid->writer_id,
                'writer_name' => $writerName,
                'order_amount' => $this->order->net_amount,
                'bid_amount' => $this->bid->bid_amount,
                'profit' => $profit,
                'profit_percentage' => $profitPercentage,
                'completion_time' => $this->bid->completion_time
            ]
        );
    }
}
