<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\Payment;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class PaymentFailedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $payment;
    protected $errorMessage;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param Payment|null $payment
     * @param string $errorMessage
     */
    public function __construct(Order $order, ?Payment $payment = null, string $errorMessage = 'Unknown error')
    {
        $this->order = $order;
        $this->payment = $payment;
        $this->errorMessage = $errorMessage;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Format amount for display if payment exists
        $amount = $this->payment ? ('$' . number_format($this->payment->amount, 2)) : 'N/A';

        // Format payment method for display
        $paymentMethod = $this->payment
            ? ucfirst(str_replace('_', ' ', $this->payment->payment_method))
            : 'selected payment method';

        // Create mail message with appropriate error info
        $mailMessage = (new MailMessage)
            ->subject('Payment Failed - Order #' . $this->order->order_number)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('We were unable to process your payment for Order #' . $this->order->order_number . '.')
            ->line('Payment Details:')
            ->line('Amount: ' . $amount)
            ->line('Payment Method: ' . $paymentMethod);

        // Add transaction ID if available
        if ($this->payment && $this->payment->transaction_id) {
            $mailMessage->line('Transaction ID: ' . $this->payment->transaction_id);
        }

        // Add error information (cleaned for user display)
        $userFriendlyError = $this->getUserFriendlyError($this->errorMessage);
        $mailMessage->line('Error: ' . $userFriendlyError);

        // Add action and support information
        return $mailMessage
            ->line('Please try again with a different payment method or contact your bank for more information.')
            ->action('Try Payment Again', url('/orders/' . $this->order->id . '/payment'))
            ->line('If you continue to experience issues, please contact our support team for assistance.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Create try again action
        $tryAgainAction = $this->createAction(
            'Try Again',
            url('/orders/' . $this->order->id . '/payment'),
            'GET',
            'primary'
        );

        // Create contact support action
        $contactSupportAction = $this->createAction(
            'Contact Support',
            url('/support/tickets/create?order_id=' . $this->order->id),
            'GET',
            'secondary'
        );

        // User-friendly error message
        $userFriendlyError = $this->getUserFriendlyError($this->errorMessage);

        // Payment method info
        $paymentMethod = $this->payment ? $this->payment->payment_method : null;

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'payment_failed',
            'Payment Failed',
            'We were unable to process your payment for Order #' . $this->order->order_number . '. ' . $userFriendlyError,
            'alert-circle',
            'danger',
            'high',
            'payment',
            [$tryAgainAction, $contactSupportAction],
            now()->addDays(7), // Expires after 7 days (shorter than success as needs action)
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'payment_id' => $this->payment ? $this->payment->id : null,
                'payment_method' => $paymentMethod,
                'amount' => $this->payment ? $this->payment->amount : null,
                'error_message' => $this->errorMessage,
                'failed_at' => now()->toISOString(),
                'order_title' => $this->order->title,
            ]
        );
    }

    /**
     * Convert technical error messages to user-friendly versions.
     *
     * @param string $errorMessage
     * @return string
     */
    private function getUserFriendlyError(string $errorMessage): string
    {
        // Map of common technical errors to user-friendly messages
        $errorMap = [
            'insufficient_funds' => 'Your card has insufficient funds.',
            'card_declined' => 'Your card was declined by the bank.',
            'expired_card' => 'Your card has expired.',
            'invalid_cvc' => 'The security code (CVC) is invalid.',
            'processing_error' => 'An error occurred while processing your card.',
            'rate_limit' => 'Too many payment attempts. Please try again later.',
        ];

        // Check if the error message contains any known technical errors
        foreach ($errorMap as $technical => $friendly) {
            if (stripos($errorMessage, $technical) !== false) {
                return $friendly;
            }
        }

        // Default user-friendly message if no specific match
        return 'Your payment could not be processed. Please try again or use a different payment method.';
    }
}
