<?php

namespace App\Notifications;

use App\Models\OrderRevision;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class RevisionAssignedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $revision;

    /**
     * Create a new notification instance.
     */
    public function __construct(OrderRevision $revision)
    {
        $this->revision = $revision;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $order = $this->revision->order;
        
        return (new MailMessage)
            ->subject("Revision Assigned - Order #{$order->order_number}")
            ->greeting("Hello {$notifiable->name},")
            ->line("You have been assigned a revision for Order #{$order->order_number}.")
            ->line("Revision Type: " . ucfirst($this->revision->revision_type))
            ->line("Reason: {$this->revision->revision_reason}")
            ->line("Due Date: " . $this->revision->due_date->format('M d, Y \a\t g:i A'))
            ->action('View Revision Details', url("/writer/revisions/{$this->revision->id}"))
            ->line('Please complete this revision by the due date to avoid any penalties.')
            ->salutation('Best regards, Academic Scribe Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $order = $this->revision->order;
        
        return [
            'type' => 'revision_assigned',
            'title' => "Revision Assigned - Order #{$order->order_number}",
            'message' => "You have been assigned a revision for Order #{$order->order_number}",
            'order_id' => $order->id,
            'order_number' => $order->order_number,
            'revision_id' => $this->revision->id,
            'revision_type' => $this->revision->revision_type,
            'due_date' => $this->revision->due_date,
            'revision_reason' => $this->revision->revision_reason,
            'timestamp' => now(),
        ];
    }
} 