<?php

namespace App\Notifications;

use App\Models\OrderRevision;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class RevisionCompletedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected OrderRevision $revision;

    /**
     * Create a new notification instance.
     */
    public function __construct(OrderRevision $revision)
    {
        $this->revision = $revision;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('revision_completed', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $order = $this->revision->order;
        $writer = $this->revision->assignedTo;
        
        return (new MailMessage)
            ->subject("Revision Completed - Order #{$order->order_number}")
            ->greeting("Hello {$notifiable->name},")
            ->line("A revision for your order has been completed by the assigned writer.")
            ->line("**Order Details:**")
            ->line("Order Number: {$order->order_number}")
            ->line("Order Title: {$order->title}")
            ->line("Writer: {$writer->name}")
            ->line("Revision Type: " . ucfirst($this->revision->revision_type))
            ->line("Revision Number: {$this->revision->revision_number}")
            ->line("Completed At: " . $this->revision->completed_at->format('M d, Y \a\t g:i A'))
            ->action('View Order Details', route('client.orders.show', $order))
            ->line('Your requested changes have been completed.')
            ->line('The revision is now under review by our quality assurance team.')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        $order = $this->revision->order;
        $writer = $this->revision->assignedTo;

        // Create view order action
        $action = $this->createAction(
            'View Order',
            route('client.orders.show', $order),
            'GET',
            'success'
        );

        return $this->getDatabaseNotificationData(
            'revision_completed',
            "Revision Completed",
            "A revision for order #{$order->order_number} has been completed by {$writer->name}.",
            'check-circle',
            'success',
            'high',
            'order',
            [$action],
            null,
            [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'title' => $order->title,
                'revision_id' => $this->revision->id,
                'revision_number' => $this->revision->revision_number,
                'revision_type' => $this->revision->revision_type,
                'writer_id' => $writer->id,
                'writer_name' => $writer->name,
                'completed_at' => $this->revision->completed_at->toISOString(),
                'completion_notes' => $this->revision->admin_notes
            ]
        );
    }
} 