<?php

namespace App\Notifications;

use App\Models\OrderRevision;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class RevisionStartedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected OrderRevision $revision;

    /**
     * Create a new notification instance.
     */
    public function __construct(OrderRevision $revision)
    {
        $this->revision = $revision;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('revision_started', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $order = $this->revision->order;
        $writer = $this->revision->assignedTo;
        
        return (new MailMessage)
            ->subject("Revision Started - Order #{$order->order_number}")
            ->greeting("Hello {$notifiable->name},")
            ->line("A revision for your order has been started by the assigned writer.")
            ->line("**Order Details:**")
            ->line("Order Number: {$order->order_number}")
            ->line("Order Title: {$order->title}")
            ->line("Writer: {$writer->name}")
            ->line("Revision Type: " . ucfirst($this->revision->revision_type))
            ->line("Revision Number: {$this->revision->revision_number}")
            ->line("Started At: " . $this->revision->started_at->format('M d, Y \a\t g:i A'))
            ->action('View Order Details', route('client.orders.show', $order))
            ->line('The writer is now working on your requested changes.')
            ->line('You will be notified when the revision is completed.')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        $order = $this->revision->order;
        $writer = $this->revision->assignedTo;

        // Create view order action
        $action = $this->createAction(
            'View Order',
            route('client.orders.show', $order),
            'GET',
            'info'
        );

        return $this->getDatabaseNotificationData(
            'revision_started',
            "Revision Started",
            "A revision for order #{$order->order_number} has been started by {$writer->name}.",
            'play-circle',
            'info',
            'medium',
            'order',
            [$action],
            null,
            [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'title' => $order->title,
                'revision_id' => $this->revision->id,
                'revision_number' => $this->revision->revision_number,
                'revision_type' => $this->revision->revision_type,
                'writer_id' => $writer->id,
                'writer_name' => $writer->name,
                'started_at' => $this->revision->started_at->toISOString(),
                'due_date' => $this->revision->due_date ? $this->revision->due_date->toISOString() : null
            ]
        );
    }
} 