<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\OrderFile;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WorkSubmittedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $uploadedFiles;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param array $uploadedFiles
     */
    public function __construct(Order $order, array $uploadedFiles = [])
    {
        $this->order = $order;
        $this->uploadedFiles = $uploadedFiles;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('work_submitted', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $userName = $notifiable->name ?? 'Client';
        $writerName = $this->order->writer ? $this->order->writer->name : 'Your assigned writer';
        $fileCount = count($this->uploadedFiles);
        $submissionTime = $this->order->work_submitted_at ? $this->order->work_submitted_at->format('M j, Y g:i A') : 'Just now';
        $deadline = $this->order->deadline ? $this->order->deadline->format('M j, Y g:i A') : 'Not set';
        $isLateSubmission = $this->order->work_submitted_at && $this->order->deadline && 
                           $this->order->work_submitted_at->isAfter($this->order->deadline);

        $mailMessage = (new MailMessage)
            ->subject("Work Submitted for Order #{$this->order->order_number}")
            ->greeting("Hello {$userName},")
            ->line("Great news! Your writer has submitted the completed work for your order #{$this->order->order_number} ({$this->order->title}).")
            ->line("Order Details:")
            ->line("- Order #: {$this->order->order_number}")
            ->line("- Title: {$this->order->title}")
            ->line("- Writer: {$writerName}")
            ->line("- Submission Time: {$submissionTime}")
            ->line("- Client Deadline: {$deadline}")
            ->line("- Files submitted: {$fileCount} file(s)");

        if ($isLateSubmission) {
            $hoursLate = $this->order->work_submitted_at->diffInHours($this->order->deadline);
            $mailMessage->line("⚠️ Note: This submission was made {$hoursLate} hour(s) after your deadline.")
                ->line("Our admin team will review the work quality and assess any applicable penalties.");
        } else {
            $mailMessage->line("✅ Work was submitted on time! Great job by your writer.");
        }

        if (!empty($this->uploadedFiles)) {
            $mailMessage->line("Submitted files:");
            foreach ($this->uploadedFiles as $file) {
                $mailMessage->line("- {$file->file_name}");
            }
        }

        $mailMessage->line("Our admin team will now review the work and upload it to your account within 24 hours.")
            ->line("You'll receive another notification once the work is ready for your review.")
            ->action('View Order Details', url("/orders/{$this->order->id}"))
            ->line('Thank you for choosing our service!');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $writerName = $this->order->writer ? $this->order->writer->name : 'Your writer';
        $fileCount = count($this->uploadedFiles);
        $submissionTime = $this->order->work_submitted_at ? $this->order->work_submitted_at->format('M j, Y g:i A') : 'Just now';
        $deadline = $this->order->deadline ? $this->order->deadline->format('M j, Y g:i A') : 'Not set';
        $isLateSubmission = $this->order->work_submitted_at && $this->order->deadline && 
                           $this->order->work_submitted_at->isAfter($this->order->deadline);

        // Create view order action
        $action = $this->createAction(
            'View Order',
            url("/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        $statusMessage = $isLateSubmission 
            ? "{$writerName} has submitted LATE work for your order #{$this->order->order_number}. Admin will review and assess penalties."
            : "{$writerName} has submitted the completed work for your order #{$this->order->order_number}. Admin will review and upload within 24 hours.";

        return $this->getDatabaseNotificationData(
            'work_submitted',
            "Work Submitted",
            $statusMessage,
            'upload',
            $isLateSubmission ? 'warning' : 'info',
            'high',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'writer_name' => $writerName,
                'file_count' => $fileCount,
                'submission_time' => $this->order->work_submitted_at ? $this->order->work_submitted_at->toISOString() : now()->toISOString(),
                'client_deadline' => $this->order->deadline ? $this->order->deadline->toISOString() : null,
                'is_late_submission' => $isLateSubmission,
                'hours_late' => $isLateSubmission && $this->order->work_submitted_at && $this->order->deadline ? 
                    $this->order->work_submitted_at->diffInHours($this->order->deadline) : 0,
                'files' => $this->uploadedFiles->map(function ($file) {
                    return [
                        'name' => $file->file_name,
                        'size' => $file->file_size,
                        'type' => $file->file_type
                    ];
                })->toArray()
            ]
        );
    }
}
