<?php

namespace App\Notifications;

use App\Models\WriterFine;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterFineAppealed extends Notification implements ShouldQueue
{
    use Queueable;

    protected $fine;
    protected $appealReason;

    /**
     * Create a new notification instance.
     */
    public function __construct(WriterFine $fine, string $appealReason)
    {
        $this->fine = $fine;
        $this->appealReason = $appealReason;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Fine Appeal Submitted - Writer #' . $this->fine->writer_id)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('A writer has submitted an appeal for a fine.')
            ->line('**Fine Details:**')
            ->line('• **Fine ID:** #' . $this->fine->id)
            ->line('• **Writer:** ' . $this->fine->writer->name . ' (ID: ' . $this->fine->writer_id . ')')
            ->line('• **Fine Amount:** $' . number_format($this->fine->fine_amount, 2) . ' ' . $this->fine->currency)
            ->line('• **Fine Type:** ' . ucwords(str_replace('_', ' ', $this->fine->fine_type)))
            ->line('• **Fine Reason:** ' . $this->fine->fine_reason)
            ->line('• **Applied:** ' . $this->fine->applied_at->format('M d, Y \a\t g:i A'))
            ->line('**Appeal Details:**')
            ->line('• **Appeal Reason:** ' . $this->appealReason)
            ->line('• **Submitted:** ' . now()->format('M d, Y \a\t g:i A'))
            ->when($this->fine->order_id, function ($message) {
                return $message->line('• **Related Order:** #' . $this->fine->order_id);
            })
            ->when($this->fine->infraction_id, function ($message) {
                return $message->line('• **Related Infraction:** #' . $this->fine->infraction_id);
            })
            ->line('**Action Required:**')
            ->line('Please review this appeal and make a decision within 48 hours.')
            ->action('Review Appeal', route('admin.fines.appeals.show', $this->fine->id))
            ->line('**Note:** The fine is temporarily suspended until the appeal is resolved.')
            ->salutation('Best regards, AcademicScribe System');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'fine_id' => $this->fine->id,
            'writer_id' => $this->fine->writer_id,
            'writer_name' => $this->fine->writer->name,
            'fine_amount' => $this->fine->fine_amount,
            'fine_type' => $this->fine->fine_type,
            'fine_reason' => $this->fine->fine_reason,
            'appeal_reason' => $this->appealReason,
            'appeal_submitted_at' => now(),
            'order_id' => $this->fine->order_id,
            'infraction_id' => $this->fine->infraction_id,
            'type' => 'writer_fine_appealed',
            'message' => 'Writer ' . $this->fine->writer->name . ' has appealed fine #' . $this->fine->id,
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 