<?php

namespace App\Notifications;

use App\Models\WriterFine;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterFineApplied extends Notification implements ShouldQueue
{
    use Queueable;

    protected $fine;

    /**
     * Create a new notification instance.
     */
    public function __construct(WriterFine $fine)
    {
        $this->fine = $fine;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $applicationMethods = [
            'immediate_deduction' => 'Immediate Deduction',
            'future_earning_deduction' => 'Future Earning Deduction',
            'withdrawal_deduction' => 'Withdrawal Deduction',
            'bonus_cancellation' => 'Bonus Cancellation',
            'other' => 'Other'
        ];

        $fineTypes = [
            'fixed_amount' => 'Fixed Amount',
            'percentage_based' => 'Percentage Based',
            'progressive' => 'Progressive',
            'opportunity_cost' => 'Opportunity Cost',
            'quality_penalty' => 'Quality Penalty',
            'deadline_violation' => 'Deadline Violation',
            'other' => 'Other'
        ];

        return (new MailMessage)
            ->subject('Writer Fine Applied - $' . number_format($this->fine->fine_amount, 2))
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('A fine has been applied to your account.')
            ->line('**Fine Details:**')
            ->line('• **Amount:** $' . number_format($this->fine->fine_amount, 2) . ' ' . $this->fine->currency)
            ->line('• **Type:** ' . ($fineTypes[$this->fine->fine_type] ?? $this->fine->fine_type))
            ->line('• **Application Method:** ' . ($applicationMethods[$this->fine->application_method] ?? $this->fine->application_method))
            ->line('• **Reason:** ' . $this->fine->fine_reason)
            ->line('• **Applied:** ' . $this->fine->applied_at->format('M d, Y \a\t g:i A'))
            ->when($this->fine->effective_from, function ($message) {
                return $message->line('• **Effective From:** ' . $this->fine->effective_from->format('M d, Y'));
            })
            ->when($this->fine->infraction_id, function ($message) {
                return $message->line('• **Related Infraction:** #' . $this->fine->infraction_id);
            })
            ->when($this->fine->order_id, function ($message) {
                return $message->line('• **Related Order:** #' . $this->fine->order_id);
            })
            ->line('**Fine Application:**')
            ->line('This fine will be deducted from your future earnings according to the application method specified.')
            ->line('**Next Steps:**')
            ->line('1. Review the fine details and reason')
            ->line('2. Contact support if you believe this fine was applied in error')
            ->line('3. Submit an appeal if you disagree with the fine')
            ->action('View Fine Details', route('writer.fines.show', $this->fine->id))
            ->line('**Important:** This fine may affect your account balance and future payment eligibility.')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'fine_id' => $this->fine->id,
            'fine_amount' => $this->fine->fine_amount,
            'fine_type' => $this->fine->fine_type,
            'application_method' => $this->fine->application_method,
            'fine_reason' => $this->fine->fine_reason,
            'applied_at' => $this->fine->applied_at,
            'effective_from' => $this->fine->effective_from,
            'infraction_id' => $this->fine->infraction_id,
            'order_id' => $this->fine->order_id,
            'type' => 'writer_fine_applied',
            'message' => 'A fine of $' . number_format($this->fine->fine_amount, 2) . ' has been applied to your account.',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 