<?php

namespace App\Notifications;

use App\Models\WriterFine;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterFineDeducted extends Notification implements ShouldQueue
{
    use Queueable;

    protected $fine;
    protected $deductionAmount;
    protected $paymentRecord;

    /**
     * Create a new notification instance.
     */
    public function __construct(WriterFine $fine, float $deductionAmount, $paymentRecord = null)
    {
        $this->fine = $fine;
        $this->deductionAmount = $deductionAmount;
        $this->paymentRecord = $paymentRecord;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('Fine Deducted - $' . number_format($this->deductionAmount, 2) . ' Applied')
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('A fine has been deducted from your earnings.')
            ->line('**Deduction Details:**')
            ->line('• **Fine Amount:** $' . number_format($this->fine->fine_amount, 2) . ' ' . $this->fine->currency)
            ->line('• **Amount Deducted:** $' . number_format($this->deductionAmount, 2) . ' ' . $this->fine->currency)
            ->line('• **Remaining Balance:** $' . number_format($this->fine->remaining_amount, 2) . ' ' . $this->fine->currency)
            ->line('• **Deduction Date:** ' . now()->format('M d, Y \a\t g:i A'))
            ->line('• **Fine Reason:** ' . $this->fine->fine_reason);

        if ($this->paymentRecord) {
            $message->line('• **Payment Record:** #' . $this->paymentRecord->id);
        }

        if ($this->fine->order_id) {
            $message->line('• **Related Order:** #' . $this->fine->order_id);
        }

        $message->line('**Fine Status:**')
            ->line('This fine has been applied to your account. The remaining balance will continue to be deducted from future earnings until fully resolved.')
            ->action('View Fine Details', route('writer.fines.show', $this->fine->id))
            ->line('**Account Impact:**')
            ->line('This deduction affects your current account balance and may impact future payment eligibility.')
            ->salutation('Best regards, AcademicScribe Team');

        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'fine_id' => $this->fine->id,
            'fine_amount' => $this->fine->fine_amount,
            'deduction_amount' => $this->deductionAmount,
            'remaining_amount' => $this->fine->remaining_amount,
            'fine_reason' => $this->fine->fine_reason,
            'deducted_at' => now(),
            'payment_record_id' => $this->paymentRecord?->id,
            'order_id' => $this->fine->order_id,
            'type' => 'writer_fine_deducted',
            'message' => 'Fine of $' . number_format($this->deductionAmount, 2) . ' has been deducted from your earnings.',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 