<?php

namespace App\Notifications;

use App\Models\WriterInfraction;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterInfractionResolved extends Notification implements ShouldQueue
{
    use Queueable;

    protected $infraction;
    protected $resolutionNotes;
    protected $resolvedBy;

    /**
     * Create a new notification instance.
     */
    public function __construct(WriterInfraction $infraction, string $resolutionNotes, $resolvedBy)
    {
        $this->infraction = $infraction;
        $this->resolutionNotes = $resolutionNotes;
        $this->resolvedBy = $resolvedBy;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Infraction Resolved - Case #' . $this->infraction->id)
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('Your infraction case has been resolved.')
            ->line('**Resolution Details:**')
            ->line('• **Infraction ID:** #' . $this->infraction->id)
            ->line('• **Type:** ' . ucwords(str_replace('_', ' ', $this->infraction->infraction_type)))
            ->line('• **Severity:** ' . ucfirst($this->infraction->severity_level))
            ->line('• **Resolved By:** ' . $this->resolvedBy->name)
            ->line('• **Resolution Date:** ' . now()->format('M d, Y \a\t g:i A'))
            ->line('• **Resolution Notes:** ' . $this->resolutionNotes)
            ->when($this->infraction->order_id, function ($message) {
                return $message->line('• **Related Order:** #' . $this->infraction->order_id);
            })
            ->line('**Case Status:**')
            ->line('This infraction case has been closed and resolved. All associated disciplinary actions have been completed.')
            ->line('**Account Impact:**')
            ->line('Your account status has been updated based on the resolution of this case.')
            ->when($this->infraction->fines->count() > 0, function ($message) {
                $totalFines = $this->infraction->fines->sum('fine_amount');
                return $message->line('**Financial Impact:**')
                    ->line('Total fines applied: $' . number_format($totalFines, 2));
            })
            ->line('**Next Steps:**')
            ->line('1. Review the resolution details')
            ->line('2. Ensure compliance with platform policies going forward')
            ->line('3. Contact support if you have questions about the resolution')
            ->action('View Infraction Details', route('writer.infractions.show', $this->infraction->id))
            ->line('**Moving Forward:**')
            ->line('We encourage you to maintain high standards and avoid future infractions. Your continued success on our platform is important to us.')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'infraction_id' => $this->infraction->id,
            'infraction_type' => $this->infraction->infraction_type,
            'severity_level' => $this->infraction->severity_level,
            'resolution_notes' => $this->resolutionNotes,
            'resolved_by' => $this->resolvedBy->name,
            'resolved_at' => now(),
            'order_id' => $this->infraction->order_id,
            'total_fines' => $this->infraction->fines->sum('fine_amount'),
            'type' => 'writer_infraction_resolved',
            'message' => 'Your infraction case has been resolved.',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 