<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterPerformanceImprovementPlan extends Notification implements ShouldQueue
{
    use Queueable;

    protected $pipData;
    protected $goals;
    protected $timeline;
    protected $reviewSchedule;
    protected $consequences;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        array $pipData,
        array $goals,
        string $timeline,
        array $reviewSchedule,
        array $consequences = []
    ) {
        $this->pipData = $pipData;
        $this->goals = $goals;
        $this->timeline = $timeline;
        $this->reviewSchedule = $reviewSchedule;
        $this->consequences = $consequences;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('📋 Performance Improvement Plan (PIP) - Action Required')
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('**IMPORTANT: You have been placed on a Performance Improvement Plan (PIP).**')
            ->line('**PIP Overview:**')
            ->line('• **Plan ID:** ' . ($this->pipData['plan_id'] ?? 'N/A'))
            ->line('• **Reason:** ' . ($this->pipData['reason'] ?? 'Performance concerns'))
            ->line('• **Start Date:** ' . ($this->pipData['start_date'] ?? now()->format('M d, Y')))
            ->line('• **Timeline:** ' . $this->timeline)
            ->line('• **Status:** Active')
            ->line('**Performance Goals:**')
            ->line('The following specific goals have been established for improvement:');

        // Add goals
        foreach ($this->goals as $index => $goal) {
            $message->line('**Goal ' . ($index + 1) . ':** ' . $goal['description']);
            if (isset($goal['target'])) {
                $message->line('   Target: ' . $goal['target']);
            }
            if (isset($goal['measurement'])) {
                $message->line('   Measurement: ' . $goal['measurement']);
            }
        }

        $message->line('**Review Schedule:**')
            ->line('Your progress will be reviewed at the following intervals:');

        foreach ($this->reviewSchedule as $review) {
            $message->line('• **' . $review['type'] . ':** ' . $review['date'] . ' - ' . $review['focus']);
        }

        $message->line('**Required Actions:**')
            ->line('1. **IMMEDIATE:** Acknowledge this PIP within 24 hours')
            ->line('2. **ONGOING:** Work diligently to meet all performance goals')
            ->line('3. **REGULAR:** Submit progress reports as scheduled')
            ->line('4. **COMMUNICATION:** Maintain open communication with your supervisor')
            ->line('5. **IMPROVEMENT:** Take proactive steps to address identified issues');

        if (!empty($this->consequences)) {
            $message->line('**Consequences of Non-Compliance:**');
            foreach ($this->consequences as $consequence) {
                $message->line('• ' . $consequence);
            }
        }

        $message->line('**Support Available:**')
            ->line('• Regular check-ins with your supervisor')
            ->line('• Access to training resources and materials')
            ->line('• Performance coaching and guidance')
            ->line('• Clear feedback on progress and areas for improvement');

        $message->line('**Success Criteria:**')
            ->line('To successfully complete this PIP, you must:')
            ->line('1. Meet or exceed all established performance goals')
            ->line('2. Maintain consistent improvement throughout the timeline')
            ->line('3. Demonstrate sustained positive change')
            ->line('4. Complete all required progress reports and reviews');

        $message->action('Acknowledge PIP', route('writer.pip.acknowledge', $this->pipData['plan_id'] ?? 'new'))
            ->action('View PIP Details', route('writer.pip.show', $this->pipData['plan_id'] ?? 'new'))
            ->action('Contact Supervisor', route('writer.pip.contact'))
            ->line('**Important Notes:**')
            ->line('• This PIP is designed to help you succeed and improve your performance')
            ->line('• Failure to meet goals may result in further disciplinary action')
            ->line('• Success will result in PIP completion and status restoration')
            ->line('• You have the right to appeal this PIP within 48 hours')
            ->line('**Next Steps:**')
            ->line('1. Review this PIP carefully and understand all requirements')
            ->line('2. Acknowledge receipt within 24 hours')
            ->line('3. Begin working on improvement goals immediately')
            ->line('4. Schedule your first progress review')
            ->salutation('Best regards, AcademicScribe Performance Management Team');

        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'plan_id' => $this->pipData['plan_id'] ?? null,
            'reason' => $this->pipData['reason'] ?? 'Performance concerns',
            'start_date' => $this->pipData['start_date'] ?? now(),
            'timeline' => $this->timeline,
            'goals' => $this->goals,
            'review_schedule' => $this->reviewSchedule,
            'consequences' => $this->consequences,
            'status' => 'active',
            'created_at' => now(),
            'type' => 'writer_performance_improvement_plan',
            'message' => 'You have been placed on a Performance Improvement Plan (PIP).',
            'priority' => 'high',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 