<?php

namespace App\Policies;

use App\Models\Order;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class OrderPolicy
{
    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        // All authenticated users can view their own orders
        return true;
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Order $order): bool
    {
        // Clients can view their own orders
        if ($user->user_type === 'client') {
            return $user->id == $order->user_id;  // Fixed: === to ==
        }

        // Writers can view orders assigned to them
        if ($user->user_type === 'writer') {
            return $user->id == $order->writer_id;  // Fixed: === to ==
        }

        // Admins can view all orders
        if ($user->user_type === 'admin') {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        // Only clients can create orders
        return $user->user_type === 'client';
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Order $order): bool
    {
        // Only clients can update their own orders, and only if they're still in bidding phase
        if ($user->user_type === 'client') {
            return $user->id == $order->user_id &&  // Fixed: === to ==
                in_array($order->order_status, ['pending', 'bidding']);
        }

        // Admins can update any order
        if ($user->user_type === 'admin') {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Order $order): bool
    {
        // Only clients can delete their own orders, and only if they're still pending or in bidding
        if ($user->user_type === 'client') {
            return $user->id == $order->user_id &&  // Fixed: === to ==
                in_array($order->order_status, ['pending', 'bidding']) &&
                $order->payment_status === 'unpaid';
        }

        // Admins can delete any order
        if ($user->user_type === 'admin') {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Order $order): bool
    {
        // Only admins can restore orders
        return $user->user_type === 'admin';
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Order $order): bool
    {
        // Only admins can permanently delete orders
        return $user->user_type === 'admin';
    }

    /**
     * Determine whether the user can pay for the order.
     */
    public function pay(User $user, Order $order): bool
    {
        // Only clients who own the order can pay for it, and only if it's unpaid
        if ($user->user_type === 'client') {
            return $user->id == $order->user_id && $order->payment_status === 'unpaid';  // Fixed: === to ==
        }

        return false;
    }

    /**
     * Determine whether the user can view payment information for the order.
     */
    public function viewPayments(User $user, Order $order): bool
    {
        // Clients can view payment info for their own orders
        if ($user->user_type === 'client') {
            return $user->id == $order->user_id;  // Fixed: === to ==
        }

        // Admins can view payment info for all orders
        if ($user->user_type === 'admin') {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can upload files to the order.
     */
    public function uploadFiles(User $user, Order $order): bool
    {
        // Clients can upload files to their own orders
        if ($user->user_type === 'client') {
            return $user->id == $order->user_id;  // Fixed: === to ==
        }

        // Writers can upload files to orders assigned to them
        if ($user->user_type === 'writer') {
            return $user->id == $order->writer_id;  // Fixed: === to ==
        }

        // Admins can upload files to any order
        if ($user->user_type === 'admin') {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can download files from the order.
     */
    public function downloadFiles(User $user, Order $order): bool
    {
        // Clients can download files from their own orders
        if ($user->user_type === 'client') {
            return $user->id == $order->user_id;  // Fixed: === to ==
        }

        // Writers can download files from orders assigned to them
        if ($user->user_type === 'writer') {
            return $user->id == $order->writer_id ||  // Fixed: === to ==
                $user->id == $order->user_id;     // Fixed: === to ==
        }

        // Admins can download files from any order
        if ($user->user_type === 'admin') {
            return true;
        }

        return false;
    }
}
