<?php

namespace App\Repositories;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Query\Builder;
use Illuminate\Pagination\LengthAwarePaginator;

class NotificationRepository
{
    /**
     * The notifications table name.
     *
     * @var string
     */
    protected $table = 'notifications';

    /**
     * Get notifications for a specific user with filtering options.
     *
     * @param int $userId
     * @param array $filters
     * @return LengthAwarePaginator
     */
    public function getForUser(int $userId, array $filters = []): LengthAwarePaginator
    {
        try {
            $query = DB::table($this->table)
                ->where('notifiable_id', $userId)
                ->where('notifiable_type', User::class);

            // Apply filters
            $this->applyFilters($query, $filters);

            // Default sort is newest first
            $sortField = $filters['sort_field'] ?? 'created_at';
            $sortDirection = $filters['sort_direction'] ?? 'desc';
            $query->orderBy($sortField, $sortDirection);

            // Pagination
            $perPage = $filters['per_page'] ?? 15;
            $page = $filters['page'] ?? 1;

            return $query->paginate($perPage, ['*'], 'page', $page);
        } catch (Exception $e) {
            Log::error("Error retrieving notifications for user {$userId}", [
                'exception' => $e->getMessage(),
                'filters' => $filters
            ]);

            // Return empty paginator rather than throwing exception
            return new LengthAwarePaginator(
                [],
                0,
                $filters['per_page'] ?? 15,
                $filters['page'] ?? 1
            );
        }
    }

    /**
     * Mark specific notifications as read.
     *
     * @param array $notificationIds
     * @return bool
     */
    public function markAsRead(array $notificationIds): bool
    {
        if (empty($notificationIds)) {
            return false;
        }

        try {
            $affected = DB::table($this->table)
                ->whereIn('id', $notificationIds)
                ->whereNull('read_at')
                ->update(['read_at' => now()]);

            return $affected > 0;
        } catch (Exception $e) {
            Log::error("Error marking notifications as read", [
                'exception' => $e->getMessage(),
                'notification_ids' => $notificationIds
            ]);
            return false;
        }
    }

    /**
     * Mark all notifications as read for a specific user.
     *
     * @param int $userId
     * @return bool
     */
    public function markAllAsReadForUser(int $userId): bool
    {
        try {
            $affected = DB::table($this->table)
                ->where('notifiable_id', $userId)
                ->where('notifiable_type', User::class)
                ->whereNull('read_at')
                ->update(['read_at' => now()]);

            return $affected > 0;
        } catch (Exception $e) {
            Log::error("Error marking all notifications as read for user {$userId}", [
                'exception' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get unread notification count for a user.
     *
     * @param int $userId
     * @return int
     */
    public function getUnreadCount(int $userId): int
    {
        try {
            return DB::table($this->table)
                ->where('notifiable_id', $userId)
                ->where('notifiable_type', User::class)
                ->whereNull('read_at')
                ->count();
        } catch (Exception $e) {
            Log::error("Error getting unread notification count for user {$userId}", [
                'exception' => $e->getMessage()
            ]);
            return 0;
        }
    }

    /**
     * Purge old notifications.
     *
     * @param int $daysToKeep
     * @return int Number of deleted records
     */
    public function purgeOld(int $daysToKeep = 90): int
    {
        try {
            $cutoffDate = now()->subDays($daysToKeep);

            // Delete old read notifications
            $deleted = DB::table($this->table)
                ->whereNotNull('read_at')
                ->where('created_at', '<', $cutoffDate)
                ->delete();

            Log::info("Purged {$deleted} old notifications older than {$daysToKeep} days");

            return $deleted;
        } catch (Exception $e) {
            Log::error("Error purging old notifications", [
                'exception' => $e->getMessage(),
                'days_to_keep' => $daysToKeep
            ]);
            return 0;
        }
    }

    /**
     * Store a new notification.
     *
     * @param int $notifiableId
     * @param string $notifiableType
     * @param array $data
     * @return string|null The ID of the created notification, or null on failure
     */
    public function store(int $notifiableId, string $notifiableType, array $data): ?string
    {
        try {
            $id = (string) Str::uuid();

            DB::table($this->table)->insert([
                'id' => $id,
                'type' => $data['type'] ?? null,
                'notifiable_id' => $notifiableId,
                'notifiable_type' => $notifiableType,
                'data' => json_encode($data),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return $id;
        } catch (Exception $e) {
            Log::error("Error storing notification", [
                'exception' => $e->getMessage(),
                'notifiable_id' => $notifiableId,
                'notifiable_type' => $notifiableType
            ]);
            return null;
        }
    }

    /**
     * Delete specific notifications.
     *
     * @param array $notificationIds
     * @return bool
     */
    public function delete(array $notificationIds): bool
    {
        if (empty($notificationIds)) {
            return false;
        }

        try {
            $deleted = DB::table($this->table)
                ->whereIn('id', $notificationIds)
                ->delete();

            return $deleted > 0;
        } catch (Exception $e) {
            Log::error("Error deleting notifications", [
                'exception' => $e->getMessage(),
                'notification_ids' => $notificationIds
            ]);
            return false;
        }
    }

    /**
     * Apply filters to the notification query.
     *
     * @param Builder $query
     * @param array $filters
     * @return void
     */
    protected function applyFilters(Builder $query, array $filters): void
    {
        // Filter by read status
        if (isset($filters['read'])) {
            if ($filters['read']) {
                $query->whereNotNull('read_at');
            } else {
                $query->whereNull('read_at');
            }
        }

        // Filter by notification type
        if (!empty($filters['type'])) {
            $query->where('type', 'LIKE', "%{$filters['type']}%");
        }

        // Filter by category (extracted from data)
        if (!empty($filters['category'])) {
            $query->whereRaw("JSON_EXTRACT(data, '$.category') = ?", [$filters['category']]);
        }

        // Filter by date range
        if (!empty($filters['from_date'])) {
            $query->where('created_at', '>=', Carbon::parse($filters['from_date']));
        }

        if (!empty($filters['to_date'])) {
            $query->where('created_at', '<=', Carbon::parse($filters['to_date']));
        }

        // Filter by search term (searches in the data JSON)
        if (!empty($filters['search'])) {
            $searchTerm = $filters['search'];
            $query->where(function ($q) use ($searchTerm) {
                $q->whereRaw("JSON_EXTRACT(data, '$.title') LIKE ?", ["%{$searchTerm}%"])
                    ->orWhereRaw("JSON_EXTRACT(data, '$.message') LIKE ?", ["%{$searchTerm}%"]);
            });
        }
    }
}
