<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Notification as NotificationFacade;

class AdminNotificationService
{
    /**
     * Send a notification to all admins.
     *
     * @param mixed $notification The notification to send
     * @param bool $includeSuperAdmins Whether to include super admins
     * @return void
     */
    public static function notifyAllAdmins($notification, bool $includeSuperAdmins = true)
    {
        $adminUsers = User::whereIn('user_type', ['admin', 'super_admin'])->get();


        Log::info('Sending notification to ' . $adminUsers->count() . ' admin users');


        $query = User::where('user_type', 'admin');

        if ($includeSuperAdmins) {
            $query->orWhere('user_type', 'super_admin');
        }

        $admins = $query->get();

        NotificationFacade::send($admins, $notification);
    }

    /**
     * Send a notification only to super admins.
     *
     * @param mixed $notification The notification to send
     * @return void
     */
    public static function notifySuperAdmins($notification)
    {
        $superAdmins = User::where('user_type', 'super_admin')->get();
        NotificationFacade::send($superAdmins, $notification);
    }

    /**
     * Send a notification to a specific admin.
     *
     * @param User $admin The admin to notify
     * @param mixed $notification The notification to send
     * @return void
     */
    public static function notifyAdmin(User $admin, $notification)
    {
        $admin->notify($notification);
    }

    /**
     * Get the system admin user (for legacy compatibility).
     *
     * @return \App\Notifications\AdminNotifiable
     */
    public static function getSystemAdminNotifiable()
    {
        $settings = \App\Models\AdminSetting::first();
        $adminEmail = $settings ? $settings->admin_email : config('mail.from.address');

        return new \App\Notifications\AdminNotifiable($adminEmail);
    }
}
