<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use App\Models\BlogTag;
use App\Models\BlogPost;
use App\Models\BlogAuthor;
use Illuminate\Support\Str;
use App\Models\BlogCategory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BlogImportService
{
    private array $config;
    private array $importResults = [];

    public function __construct()
    {
        $this->config = config('blog_uploads');
    }

    /**
     * Import processed blog files to database
     */
    public function importProcessedFiles(string $uploadId): array
    {
        try {
            // Get processed files
            $processedFiles = $this->getProcessedFiles($uploadId);

            if (empty($processedFiles)) {
                return [
                    'success' => false,
                    'error' => 'No processed files found for upload ID: ' . $uploadId,
                ];
            }

            $results = [];

            DB::beginTransaction();

            // First pass: Create all blog posts
            $createdPosts = [];
            foreach ($processedFiles as $file) {
                $result = $this->importSingleFile($file, $uploadId);
                $results[] = $result;

                // Stop processing if any file fails
                if (!$result['success']) {
                    DB::rollBack();
                    return [
                        'success' => false,
                        'error' => 'Import failed for file: ' . $file['filename'],
                        'details' => $result,
                    ];
                }

                // Store created post for related posts processing
                if (isset($result['blog_post_id'])) {
                    $createdPosts[$result['blog_post_slug']] = $result['blog_post_id'];
                }
            }

            // Second pass: Process related posts now that all posts exist
            foreach ($processedFiles as $index => $file) {
                $this->processRelatedPosts($file, $createdPosts, $results[$index]);
            }

            DB::commit();

            // Log successful import
            Log::info('Blog files imported successfully', [
                'upload_id' => $uploadId,
                'files_imported' => count($results),
                'results' => $results,
            ]);

            return [
                'success' => true,
                'upload_id' => $uploadId,
                'imported_files' => $results,
                'total_imported' => count($results),
            ];
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Blog import failed', [
                'upload_id' => $uploadId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return [
                'success' => false,
                'error' => 'Import failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Import single processed file
     */
    private function importSingleFile(array $fileData, string $uploadId): array
    {
        try {
            $blogData = $fileData['blog_data'];

            // Validate data before import
            $validation = $this->validateImportData($blogData);
            if (!$validation['valid']) {
                return [
                    'success' => false,
                    'filename' => $fileData['filename'],
                    'errors' => $validation['errors'],
                ];
            }

            // Process category
            $category = $this->processCategory($blogData['category']);

            // Process author
            $author = $this->processAuthor($blogData['author']);

            // Process tags
            $tags = $this->processTags($blogData['tags'] ?? []);

            // Create blog post
            $blogPost = $this->createBlogPost($blogData, $category, $author);

            // Attach tags
            if (!empty($tags)) {
                $blogPost->tags()->attach($tags);
            }

            return [
                'success' => true,
                'filename' => $fileData['filename'],
                'blog_post_id' => $blogPost->id,
                'blog_post_slug' => $blogPost->slug,
                'category' => $category->name,
                'author' => $author->name,
                'tags_count' => count($tags),
                'related_posts' => $blogData['related_posts'] ?? [], // Store for later processing
                'imported_at' => Carbon::now()->toISOString(),
            ];
        } catch (\Exception $e) {
            Log::error('Single file import failed', [
                'filename' => $fileData['filename'] ?? 'unknown',
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'filename' => $fileData['filename'] ?? 'unknown',
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Process related posts relationships
     */
    private function processRelatedPosts(array $fileData, array $createdPosts, array &$importResult): void
    {
        try {
            $blogData = $fileData['blog_data'];
            $relatedPostSlugs = $blogData['related_posts'] ?? [];

            if (empty($relatedPostSlugs)) {
                return;
            }

            $currentPostId = $importResult['blog_post_id'];
            $currentPost = BlogPost::find($currentPostId);

            if (!$currentPost) {
                Log::warning('Could not find current post for related posts processing', [
                    'post_id' => $currentPostId,
                    'filename' => $fileData['filename'],
                ]);
                return;
            }

            $relatedPostIds = [];
            $foundSlugs = [];
            $notFoundSlugs = [];

            foreach ($relatedPostSlugs as $slug) {
                // First check if it's in the current batch of created posts
                if (isset($createdPosts[$slug])) {
                    $relatedPostIds[] = $createdPosts[$slug];
                    $foundSlugs[] = $slug;
                } else {
                    // Check if it exists in the database
                    $existingPost = BlogPost::where('slug', $slug)->first();
                    if ($existingPost) {
                        $relatedPostIds[] = $existingPost->id;
                        $foundSlugs[] = $slug;
                    } else {
                        $notFoundSlugs[] = $slug;
                    }
                }
            }

            // Attach related posts (sync to avoid duplicates)
            if (!empty($relatedPostIds)) {
                $currentPost->relatedPosts()->sync($relatedPostIds);

                // Also create reverse relationships (bidirectional)
                foreach ($relatedPostIds as $relatedPostId) {
                    $relatedPost = BlogPost::find($relatedPostId);
                    if ($relatedPost) {
                        $relatedPost->relatedPosts()->syncWithoutDetaching([$currentPostId]);
                    }
                }
            }

            // Update import result with related posts info
            $importResult['related_posts_processed'] = [
                'total_specified' => count($relatedPostSlugs),
                'found' => count($foundSlugs),
                'not_found' => count($notFoundSlugs),
                'found_slugs' => $foundSlugs,
                'not_found_slugs' => $notFoundSlugs,
            ];

            Log::info('Related posts processed', [
                'post_slug' => $currentPost->slug,
                'related_posts_found' => count($foundSlugs),
                'related_posts_not_found' => count($notFoundSlugs),
                'not_found_slugs' => $notFoundSlugs,
            ]);
        } catch (\Exception $e) {
            Log::error('Related posts processing failed', [
                'filename' => $fileData['filename'] ?? 'unknown',
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Get processed files from storage
     */
    private function getProcessedFiles(string $uploadId): array
    {
        $processedPath = $this->config['storage']['directories']['processed'] . '/' . $uploadId;
        $files = Storage::disk($this->config['storage']['upload_disk'])->files($processedPath);

        $processedFiles = [];

        foreach ($files as $file) {
            if (str_ends_with($file, '_processed.json')) {
                $content = Storage::disk($this->config['storage']['upload_disk'])->get($file);
                $blogData = json_decode($content, true);

                if ($blogData) {
                    $processedFiles[] = [
                        'filename' => basename($file),
                        'file_path' => $file,
                        'blog_data' => $blogData,
                    ];
                }
            }
        }

        return $processedFiles;
    }

    /**
     * Validate import data
     */
    private function validateImportData(array $data): array
    {
        $errors = [];

        // Required fields validation
        $requiredFields = ['title', 'slug', 'excerpt', 'content', 'category'];
        foreach ($requiredFields as $field) {
            if (empty($data[$field])) {
                $errors[] = "Missing required field: {$field}";
            }
        }

        // Validate slug uniqueness
        if (!empty($data['slug']) && BlogPost::where('slug', $data['slug'])->exists()) {
            $errors[] = "Slug already exists: {$data['slug']}";
        }

        // Validate published date
        if (!empty($data['published_at'])) {
            try {
                Carbon::parse($data['published_at']);
            } catch (\Exception $e) {
                $errors[] = "Invalid published_at date: {$data['published_at']}";
            }
        }

        // Validate read time
        if (isset($data['read_time']) && (!is_numeric($data['read_time']) || $data['read_time'] < 0)) {
            $errors[] = "Invalid read_time value: {$data['read_time']}";
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors,
        ];
    }

    /**
     * Process category (find or create)
     */
    private function processCategory(string $categorySlug): BlogCategory
    {
        $category = BlogCategory::where('slug', $categorySlug)->first();

        if (!$category) {
            // Create new category with basic info
            $category = BlogCategory::create([
                'name' => ucwords(str_replace('-', ' ', $categorySlug)),
                'slug' => $categorySlug,
                'description' => "Articles about " . ucwords(str_replace('-', ' ', $categorySlug)),
                'color' => 'blue',
                'icon' => 'BookOpen',
            ]);
        }

        return $category;
    }

    /**
     * Process author (find or create)
     */
    private function processAuthor($authorData): BlogAuthor
    {
        // Handle both string and object author data
        if (is_string($authorData)) {
            $authorName = $authorData;
            $authorEmail = Str::slug($authorName) . '@academicscribe.com';
            $authorBio = "Expert writer and academic professional at Academic Scribe.";
            $authorAvatar = null;
            $socialLinks = [];
        } else if (is_array($authorData)) {
            $authorName = $authorData['name'] ?? 'Unknown Author';
            $authorEmail = $authorData['email'] ?? Str::slug($authorName) . '@academicscribe.com';
            $authorBio = $authorData['bio'] ?? "Expert writer and academic professional at Academic Scribe.";
            $authorAvatar = $authorData['avatar'] ?? null;
            $socialLinks = $authorData['social'] ?? [];
        } else {
            throw new \Exception('Invalid author data format');
        }

        // Find existing author by name or email
        $author = BlogAuthor::where('name', $authorName)
            ->orWhere('email', $authorEmail)
            ->first();

        if (!$author) {
            // Create new author with full data
            $author = BlogAuthor::create([
                'name' => $authorName,
                'email' => $authorEmail,
                'bio' => $authorBio,
                'avatar' => $authorAvatar,
                'social_links' => $socialLinks,
            ]);
        }

        return $author;
    }

    /**
     * Process tags (find or create)
     */
    private function processTags(array $tagNames): array
    {
        $tagIds = [];

        foreach ($tagNames as $tagName) {
            if (empty($tagName)) continue;

            $tag = BlogTag::where('name', $tagName)->first();

            if (!$tag) {
                $tag = BlogTag::create([
                    'name' => $tagName,
                    'slug' => Str::slug($tagName),
                ]);
            }

            $tagIds[] = $tag->id;
        }

        return $tagIds;
    }

    /**
     * Create blog post - FIXED TO HANDLE ALL FIELDS
     */
    private function createBlogPost(array $data, BlogCategory $category, BlogAuthor $author): BlogPost
    {
        // Debug: Log what data we're trying to save
        Log::info('Creating blog post with data fields: ' . implode(', ', array_keys($data)));

        // Debug: Log specific fields we're having issues with
        Log::info('Featured image: ' . ($data['featured_image'] ?? 'null'));
        Log::info('Featured image alt: ' . ($data['featured_image_alt'] ?? 'null'));
        Log::info('Structured data: ' . (isset($data['structured_data']) ? 'present' : 'missing'));
        Log::info('Table of contents: ' . (isset($data['table_of_contents']) ? 'present' : 'missing'));

        $blogPost = new BlogPost();

        // Basic fields
        $blogPost->title = $data['title'];
        $blogPost->subtitle = $data['subtitle'] ?? null;
        $blogPost->slug = $data['slug'];
        $blogPost->excerpt = $data['excerpt'];
        $blogPost->content = $data['content'];
        $blogPost->category_id = $category->id;
        $blogPost->author_id = $author->id;

        // Dates
        $blogPost->published_at = !empty($data['published_at'])
            ? Carbon::parse($data['published_at'])
            : Carbon::now();

        // Numeric fields
        $blogPost->read_time = $data['read_time'] ?? 5;
        $blogPost->difficulty = $data['difficulty'] ?? 'beginner';

        // Status
        $blogPost->status = 'published';

        // Featured image
        $blogPost->featured_image = $data['featured_image'] ?? null;
        $blogPost->featured_image_alt = $data['featured_image_alt'] ?? $data['title'];

        // Settings
        $blogPost->enable_comments = true;
        $blogPost->enable_ratings = true;

        // JSON fields
        $blogPost->meta_data = $data['meta_data'] ?? [];
        $blogPost->structured_data = $data['structured_data'] ?? [];
        $blogPost->table_of_contents = $data['table_of_contents'] ?? [];
        $blogPost->social_sharing = $data['social_sharing'] ?? [];
        $blogPost->analytics = $data['analytics'] ?? [];

        $blogPost->save();

        Log::info('Blog post created successfully', [
            'id' => $blogPost->id,
            'slug' => $blogPost->slug,
            'title' => $blogPost->title,
        ]);

        return $blogPost;
    }

    /**
     * Get import statistics
     */
    public function getImportStats(string $uploadId): array
    {
        try {
            $processedFiles = $this->getProcessedFiles($uploadId);

            return [
                'upload_id' => $uploadId,
                'total_files' => count($processedFiles),
                'categories' => $this->getCategoryStats($processedFiles),
                'authors' => $this->getAuthorStats($processedFiles),
                'tags' => $this->getTagStats($processedFiles),
                'related_posts' => $this->getRelatedPostsStats($processedFiles),
            ];
        } catch (\Exception $e) {
            return [
                'error' => 'Failed to get import stats: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get category statistics
     */
    private function getCategoryStats(array $files): array
    {
        $categories = [];

        foreach ($files as $file) {
            $category = $file['blog_data']['category'] ?? 'unknown';
            $categories[$category] = ($categories[$category] ?? 0) + 1;
        }

        return $categories;
    }

    /**
     * Get author statistics
     */
    private function getAuthorStats(array $files): array
    {
        $authors = [];

        foreach ($files as $file) {
            $author = $file['blog_data']['author'] ?? 'unknown';
            if (is_array($author)) {
                $author = $author['name'] ?? 'unknown';
            }
            $authors[$author] = ($authors[$author] ?? 0) + 1;
        }

        return $authors;
    }

    /**
     * Get tag statistics
     */
    private function getTagStats(array $files): array
    {
        $tags = [];

        foreach ($files as $file) {
            $fileTags = $file['blog_data']['tags'] ?? [];
            foreach ($fileTags as $tag) {
                $tags[$tag] = ($tags[$tag] ?? 0) + 1;
            }
        }

        return $tags;
    }

    /**
     * Get related posts statistics
     */
    private function getRelatedPostsStats(array $files): array
    {
        $relatedPosts = [];
        $totalRelatedPosts = 0;

        foreach ($files as $file) {
            $fileRelatedPosts = $file['blog_data']['related_posts'] ?? [];
            $totalRelatedPosts += count($fileRelatedPosts);

            foreach ($fileRelatedPosts as $relatedPost) {
                $relatedPosts[$relatedPost] = ($relatedPosts[$relatedPost] ?? 0) + 1;
            }
        }

        return [
            'total_relationships' => $totalRelatedPosts,
            'unique_referenced_slugs' => count($relatedPosts),
            'most_referenced' => $relatedPosts,
        ];
    }

    /**
     * Re-process related posts for existing uploads
     */
    public function reprocessRelatedPosts(string $uploadId): array
    {
        try {
            $processedFiles = $this->getProcessedFiles($uploadId);

            if (empty($processedFiles)) {
                return [
                    'success' => false,
                    'error' => 'No processed files found for upload ID: ' . $uploadId,
                ];
            }

            DB::beginTransaction();

            // Get all posts from this upload batch
            $createdPosts = [];
            foreach ($processedFiles as $file) {
                $blogData = $file['blog_data'];
                $post = BlogPost::where('slug', $blogData['slug'])->first();
                if ($post) {
                    $createdPosts[$post->slug] = $post->id;
                }
            }

            $results = [];

            // Process related posts for each file
            foreach ($processedFiles as $file) {
                $result = ['filename' => $file['filename']];
                $this->processRelatedPosts($file, $createdPosts, $result);
                $results[] = $result;
            }

            DB::commit();

            return [
                'success' => true,
                'upload_id' => $uploadId,
                'processed_files' => $results,
                'message' => 'Related posts reprocessed successfully',
            ];
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Related posts reprocessing failed', [
                'upload_id' => $uploadId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'error' => 'Reprocessing failed: ' . $e->getMessage(),
            ];
        }
    }
}
