<?php

namespace App\Services;

use App\Models\Coupon;
use App\Models\LoyaltyTier;
use App\Models\Order;
use App\Models\SeasonalType;
use App\Models\User;
use App\Models\UserCoupon;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CouponService
{
    /**
     * Validate if a coupon code is valid for the given order context
     *
     * @param string $code Coupon code
     * @param array $context Order context (user_id, order_total, pages, etc.)
     * @return array Response with valid status and message
     */
    public function validateCoupon(string $code, array $context): array
    {
        try {
            $coupon = Coupon::where('coupon_code', $code)
                ->active()
                ->first();

            if (!$coupon) {
                return [
                    'valid' => false,
                    'message' => 'Invalid coupon code.',
                ];
            }

            return $coupon->validateForOrder($context);
        } catch (\Exception $e) {
            Log::error('Error validating coupon: ' . $e->getMessage(), [
                'code' => $code,
                'context' => $context,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'valid' => false,
                'message' => 'An error occurred while validating the coupon.',
            ];
        }
    }

    /**
     * Calculate the discount amount for an order using a coupon
     *
     * @param Coupon $coupon
     * @param float $orderTotal
     * @param int $pages
     * @return float Discount amount
     */
    public function calculateDiscount(Coupon $coupon, float $orderTotal, int $pages = 1): float
    {
        if ($coupon->discount_type === Coupon::DISCOUNT_PERCENTAGE) {
            return round(($orderTotal * $coupon->discount_amount) / 100, 2);
        }

        // For fixed amount discounts, ensure discount doesn't exceed order total
        return min(round((float)$coupon->discount_amount, 2), $orderTotal);
    }

    /**
     * Apply a coupon to an order
     *
     * @param Order $order The order to apply the coupon to
     * @param string $couponCode The coupon code
     * @param User|null $user The user applying the coupon (optional, defaults to order user)
     * @return array Response with success status, message, and discount data
     */
    public function applyCoupon(Order $order, string $couponCode, ?User $user = null): array
    {
        // Start database transaction
        return DB::transaction(function () use ($order, $couponCode, $user) {
            try {
                $user = $user ?? $order->user;

                if (!$user) {
                    return [
                        'success' => false,
                        'message' => 'User not found.',
                    ];
                }

                $coupon = Coupon::where('coupon_code', $couponCode)
                    ->active()
                    ->validByDate()
                    ->withinUsageLimit()
                    ->first();

                if (!$coupon) {
                    return [
                        'success' => false,
                        'message' => 'Invalid or expired coupon code.',
                    ];
                }

                // Build context for validation
                $context = [
                    'user_id' => $user->id,
                    'order_total' => $order->order_amount,
                    'pages' => $order->pages,
                    'previous_orders' => $user->orders()->count(),
                ];

                $validation = $coupon->validateForOrder($context);
                if (!$validation['valid']) {
                    return [
                        'success' => false,
                        'message' => $validation['message'],
                    ];
                }

                // Calculate discount
                $discountAmount = $this->calculateDiscount($coupon, $order->order_amount, $order->pages);

                // Apply discount to order
                $netAmount = $order->order_amount - $discountAmount;

                // Update order with coupon details
                $order->coupon_code = $couponCode;
                $order->discount = $discountAmount;
                $order->net_amount = max(0, $netAmount); // Ensure net amount isn't negative
                $order->save();

                // Track coupon usage
                $this->trackCouponUsage($coupon, $user->id, $order->id, $order->order_amount, $discountAmount);

                return [
                    'success' => true,
                    'message' => 'Coupon applied successfully.',
                    'data' => [
                        'discount_type' => $coupon->discount_type,
                        'discount_amount' => $discountAmount,
                        'original_total' => $order->order_amount,
                        'new_total' => $order->net_amount,
                    ],
                ];
            } catch (\Exception $e) {
                Log::error('Error applying coupon: ' . $e->getMessage(), [
                    'order_id' => $order->id,
                    'coupon_code' => $couponCode,
                    'trace' => $e->getTraceAsString()
                ]);

                return [
                    'success' => false,
                    'message' => 'An error occurred while applying the coupon.',
                ];
            }
        });
    }

    /**
     * Track the usage of a coupon
     *
     * @param Coupon $coupon
     * @param int $userId
     * @param int|null $orderId
     * @param float $orderTotal
     * @param float $discountAmount
     * @return UserCoupon
     */
    public function trackCouponUsage(
        Coupon $coupon,
        int $userId,
        ?int $orderId = null,
        float $orderTotal = 0,
        float $discountAmount = 0
    ): UserCoupon {
        // Increment the coupon's use count
        $coupon->increment('uses_count');

        // Record the usage for this user
        return UserCoupon::create([
            'user_id' => $userId,
            'coupon_id' => $coupon->id,
            'order_id' => $orderId,
            'order_total' => $orderTotal,
            'discount_amount' => $discountAmount,
            'used_at' => now(),
        ]);
    }

    /**
     * Get a first-order coupon for a new user.
     *
     * @param \App\Models\User $user
     * @return \App\Models\Coupon|null
     */
    public function getFirstOrderCoupon(User $user): ?\App\Models\Coupon
    {
        // Check if user is eligible for first order discount
        if (!$this->isEligibleForFirstOrderDiscount($user)) {
            return null;
        }

        // Get the first-order coupon with the highest discount
        return Coupon::where('coupon_type', Coupon::TYPE_FIRST_ORDER)
            ->orWhere('is_first_order_only', true)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->orderBy('discount_amount', 'desc')
            ->first();
    }
    /**
     * Get loyalty coupons for a user based on their order count.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getUserLoyaltyCoupons(User $user): \Illuminate\Database\Eloquent\Collection
    {
        $loyaltyTier = $this->getUserLoyaltyTier($user);

        if (!$loyaltyTier) {
            return new \Illuminate\Database\Eloquent\Collection();
        }

        return Coupon::where('coupon_type', Coupon::TYPE_LOYALTY)
            ->where('loyalty_tier_id', $loyaltyTier->id)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get();
    }

    /**
     * Get volume-based coupons applicable for a given page count.
     *
     * @param int $pageCount
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getVolumeCoupons(int $pageCount): \Illuminate\Database\Eloquent\Collection
    {
        return Coupon::where('coupon_type', Coupon::TYPE_VOLUME)
            ->where('min_pages', '<=', $pageCount)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get();
    }

    /**
     * Get anniversary coupon for a user if applicable.
     *
     * @param \App\Models\User $user
     * @return \App\Models\Coupon|null
     */
    public function getAnniversaryCoupon(User $user): ?\App\Models\Coupon
    {
        // Check if today is user's registration anniversary
        $today = now();
        $registrationDate = Carbon::parse($user->created_at);

        // Check if today is the anniversary (same month and day)
        $isAnniversary = $today->month === $registrationDate->month &&
            $today->day === $registrationDate->day;

        if (!$isAnniversary) {
            return null;
        }

        // Calculate years since registration
        $years = $today->diffInYears($registrationDate);

        // Get appropriate anniversary coupon
        return Coupon::where('coupon_type', Coupon::TYPE_ANNIVERSARY)
            ->where(function ($query) use ($years) {
                $query->whereNull('min_years')
                    ->orWhere('min_years', '<=', $years);
            })
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->orderBy('min_years', 'desc') // Get the highest tier the user qualifies for
            ->first();
    }

    /**
     * Get coupons specific to a user.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getUserCoupons(User $user): \Illuminate\Database\Eloquent\Collection
    {
        // Get coupons that are specifically assigned to this user
        return Coupon::where('user_specific', true)
            ->where('referrer_user_id', $user->id)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get();
    }



    /**
     * Get all applicable coupons for a user and order context.
     *
     * @param \App\Models\User $user
     * @param array $orderContext
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getApplicableCoupons(User $user, array $orderContext): \Illuminate\Database\Eloquent\Collection
    {
        // First, get standard eligible coupons
        $eligibleCoupons = Coupon::active()
            ->validByDate()
            ->withinUsageLimit()
            ->where(function ($query) use ($user) {
                $query->whereNull('user_specific')
                    ->orWhere('user_specific', false)
                    ->orWhere('referrer_user_id', $user->id);
            })
            ->where(function ($query) use ($orderContext) {
                // Filter by minimum order amount if specified
                if (isset($orderContext['order_total']) && $orderContext['order_total'] > 0) {
                    $query->where(function ($q) use ($orderContext) {
                        $q->whereNull('min_order_amount')
                            ->orWhere('min_order_amount', '<=', $orderContext['order_total']);
                    });
                }

                // Filter by minimum pages if specified
                if (isset($orderContext['pages']) && $orderContext['pages'] > 0) {
                    $query->where(function ($q) use ($orderContext) {
                        $q->whereNull('min_pages')
                            ->orWhere('min_pages', '<=', $orderContext['pages']);
                    });
                }
            })
            ->get();

        // Filter out first-order coupons if this isn't the user's first order
        $previousOrders = $orderContext['previous_orders'] ?? $user->orders()->count();

        if ($previousOrders > 0) {
            $eligibleCoupons = $eligibleCoupons->filter(function ($coupon) {
                return !$coupon->is_first_order_only;
            });
        }

        // Get user-specific coupons
        $userCoupons = $this->getUserCoupons($user);

        // Get type-specific coupons
        $firstOrderCoupon = $previousOrders === 0 ? $this->getFirstOrderCoupon($user) : null;
        $loyaltyCoupons = $this->getUserLoyaltyCoupons($user);
        $seasonalCoupons = $this->getActiveSeasonalCoupons();
        $volumeCoupons = $this->getVolumeCoupons($orderContext['pages'] ?? 1);
        $anniversaryCoupon = $this->getAnniversaryCoupon($user);

        //  Get active flash coupons
        $flashCoupons = $this->getActiveFlashCoupons();

        // Get default coupons (always available as fallback)
        $defaultCoupons = $this->getActiveDefaultCoupons();

        // Combine all eligible coupons
        $allCoupons = $eligibleCoupons;

        // Add additional coupon types if they exist
        if ($firstOrderCoupon) {
            $allCoupons->push($firstOrderCoupon);
        }

        if ($userCoupons->isNotEmpty()) {
            $allCoupons = $allCoupons->merge($userCoupons);
        }

        if ($loyaltyCoupons->isNotEmpty()) {
            $allCoupons = $allCoupons->merge($loyaltyCoupons);
        }

        if ($seasonalCoupons->isNotEmpty()) {
            $allCoupons = $allCoupons->merge($seasonalCoupons);
        }

        if ($volumeCoupons->isNotEmpty()) {
            $allCoupons = $allCoupons->merge($volumeCoupons);
        }

        if ($anniversaryCoupon) {
            $allCoupons->push($anniversaryCoupon);
        }

        // ADD THIS: Merge flash coupons
        if ($flashCoupons->isNotEmpty()) {
            $allCoupons = $allCoupons->merge($flashCoupons);
        }

        // Add default coupons as fallback
        if ($defaultCoupons->isNotEmpty()) {
            $allCoupons = $allCoupons->merge($defaultCoupons);
        }

        // We need to return an Eloquent Collection, so convert our merged result
        // Get unique coupon IDs and fetch them in a new query
        $couponIds = $allCoupons->pluck('id')->unique()->values()->all();

        // Return an empty Eloquent Collection if no coupons were found
        if (empty($couponIds)) {
            return new \Illuminate\Database\Eloquent\Collection();
        }

        // Return a new query to get these as an Eloquent Collection
        // TEMPORARILY EXCLUDE REFERRAL COUPONS until referral logic is fully implemented
        // TODO: Re-enable referral coupons when proper referral system is implemented
        // - Track who referred whom
        // - Validate referral relationships
        // - Ensure only legitimate referrals get discounts
        return Coupon::whereIn('id', $couponIds)
            ->where('coupon_type', '!=', Coupon::TYPE_REFERRAL)
            ->get();
    }


    /**
     * Get loyalty coupon based on user's order count
     *
     * @param User $user
     * @return Coupon|null
     */
    public function getLoyaltyCoupon(User $user): ?Coupon
    {
        $orderCount = $user->orders()->where('payment_status', 'paid')->count();

        // Get the appropriate loyalty tier for this order count
        $loyaltyTier = LoyaltyTier::getTierForOrderCount($orderCount);

        if (!$loyaltyTier) {
            return null;
        }

        return Coupon::where('coupon_type', Coupon::TYPE_LOYALTY)
            ->where('loyalty_tier_id', $loyaltyTier->id)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->first();
    }



    /**
     * Get active seasonal coupons.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getActiveSeasonalCoupons(): \Illuminate\Database\Eloquent\Collection
    {
        // Find current seasonal types
        $activeSeasons = SeasonalType::getCurrentlyActive()->pluck('id');

        if ($activeSeasons->isEmpty()) {
            return new \Illuminate\Database\Eloquent\Collection();
        }

        return Coupon::where('coupon_type', Coupon::TYPE_SEASONAL)
            ->whereIn('seasonal_type_id', $activeSeasons)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get();
    }


    /**
     * Get volume discount coupon based on page count
     *
     * @param int $pages
     * @return Coupon|null
     */
    public function getVolumeCoupon(int $pages): ?Coupon
    {
        return Coupon::where('coupon_type', Coupon::TYPE_VOLUME)
            ->where('min_pages', '<=', $pages)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->orderByDesc('min_pages') // Get the highest tier the user qualifies for
            ->first();
    }

    /**
     * Get active flash discount coupons
     *
     * @return Collection
     */
    public function getActiveFlashCoupons(): Collection
    {
        return Coupon::where('coupon_type', Coupon::TYPE_FLASH)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get();
    }

    /**
     * Get active default coupons
     *
     * @return Collection
     */
    public function getActiveDefaultCoupons(): Collection
    {
        return Coupon::where('coupon_type', Coupon::TYPE_DEFAULT)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get();
    }

    /**
     * Get referral coupon for a user
     *
     * @param int $userId
     * @return Coupon|null
     */
    public function getReferralCoupon(int $userId): ?Coupon
    {
        return Coupon::where('coupon_type', Coupon::TYPE_REFERRAL)
            ->where(function ($query) use ($userId) {
                $query->where('user_specific', false)
                    ->orWhere('referrer_user_id', $userId);
            })
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->first();
    }

    /**
     * Generate a new coupon
     *
     * @param array $data Coupon data
     * @return Coupon
     */
    public function generateCoupon(array $data): Coupon
    {
        // Ensure we have a unique coupon code
        if (empty($data['coupon_code'])) {
            $data['coupon_code'] = Coupon::generateUniqueCode();
        }

        // Set defaults if not provided
        $data['is_active'] = $data['is_active'] ?? true;
        $data['uses_count'] = 0;

        return Coupon::create($data);
    }

    /**
     * Generate loyalty coupons for all tiers
     *
     * @param string $discountType
     * @param array $tierDiscounts Array of tier_id => discount
     * @param array $options Additional coupon options
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function generateLoyaltyCoupons(
        string $discountType = Coupon::DISCOUNT_PERCENTAGE,
        array $tierDiscounts = [],
        array $options = []
    ): \Illuminate\Database\Eloquent\Collection {
        $coupons = collect();

        // Get all active loyalty tiers
        $tiers = LoyaltyTier::active()->get();

        foreach ($tiers as $tier) {
            // Skip if no discount specified for this tier
            if (!isset($tierDiscounts[$tier->id]) && !isset($tierDiscounts['*'])) {
                continue;
            }

            $discountAmount = $tierDiscounts[$tier->id] ?? $tierDiscounts['*'] ?? 0;
            if ($discountAmount <= 0) {
                continue;
            }

            $couponData = [
                'coupon_type' => Coupon::TYPE_LOYALTY,
                'loyalty_tier_id' => $tier->id,
                'discount_type' => $discountType,
                'discount_amount' => $discountAmount,
                'start_date' => $options['start_date'] ?? now(),
                'expiry_date' => $options['expiry_date'] ?? null,
                'max_uses' => $options['max_uses'] ?? null,
                'coupon_code' => $options['prefix'] ?? 'LOYALTY' . $tier->id . '_' . Coupon::generateUniqueCode(4),
                'is_active' => true,
            ];

            $coupons->push($this->generateCoupon($couponData));
        }

        // Convert Support Collection to Eloquent Collection before returning
        return new \Illuminate\Database\Eloquent\Collection($coupons->all());
    }


    /**
     * Generate anniversary coupons
     *
     * @param int $yearsActive
     * @param string $discountType
     * @param float $discountAmount
     * @param array $options
     * @return Coupon
     */
    public function generateAnniversaryCoupon(
        int $yearsActive,
        string $discountType = Coupon::DISCOUNT_PERCENTAGE,
        float $discountAmount = 15.0,
        array $options = []
    ): Coupon {
        // Increase discount for long-term customers
        if ($yearsActive >= 3) {
            $discountAmount = max($discountAmount, 20.0);
        } elseif ($yearsActive >= 2) {
            $discountAmount = max($discountAmount, 17.5);
        }

        $couponData = [
            'coupon_type' => Coupon::TYPE_ANNIVERSARY,
            'discount_type' => $discountType,
            'discount_amount' => $discountAmount,
            'start_date' => $options['start_date'] ?? now(),
            'expiry_date' => $options['expiry_date'] ?? now()->addDays(7), // Valid for a week
            'max_uses' => $options['max_uses'] ?? 1, // Usually one-time use
            'coupon_code' => $options['prefix'] ?? 'ANNIV' . $yearsActive . '_' . Coupon::generateUniqueCode(4),
            'is_active' => true,
        ];

        return $this->generateCoupon($couponData);
    }

    /**
     * Generate seasonal coupons for all active seasonal types
     *
     * @param string $discountType
     * @param array $seasonalDiscounts Array of seasonal_type_id => discount
     * @param array $options Additional coupon options
     * @return Collection
     */
    public function generateSeasonalCoupons(
        string $discountType = Coupon::DISCOUNT_PERCENTAGE,
        array $seasonalDiscounts = [],
        array $options = []
    ): Collection {
        $coupons = collect();

        // Get all active seasonal types
        $seasonalTypes = SeasonalType::active()->get();

        foreach ($seasonalTypes as $seasonalType) {
            // Skip if no discount specified for this type
            if (!isset($seasonalDiscounts[$seasonalType->id]) && !isset($seasonalDiscounts['*'])) {
                continue;
            }

            $discountAmount = $seasonalDiscounts[$seasonalType->id] ?? $seasonalDiscounts['*'] ?? 0;
            if ($discountAmount <= 0) {
                continue;
            }

            // Determine date range based on seasonal type's default dates for current year
            $startDate = $options['start_date'] ?? null;
            $expiryDate = $options['expiry_date'] ?? null;

            if (!$startDate && $seasonalType->default_start_date) {
                $startDate = Carbon::createFromFormat(
                    'Y-m-d',
                    Carbon::now()->year . '-' . $seasonalType->default_start_date->format('m-d')
                );
            }

            if (!$expiryDate && $seasonalType->default_end_date) {
                $expiryDate = Carbon::createFromFormat(
                    'Y-m-d',
                    Carbon::now()->year . '-' . $seasonalType->default_end_date->format('m-d')
                );

                // If end date is earlier than start date, it spans across years
                if ($expiryDate->lt($startDate)) {
                    $expiryDate->addYear();
                }
            }

            $couponData = [
                'coupon_type' => Coupon::TYPE_SEASONAL,
                'seasonal_type_id' => $seasonalType->id,
                'discount_type' => $discountType,
                'discount_amount' => $discountAmount,
                'start_date' => $startDate,
                'expiry_date' => $expiryDate,
                'max_uses' => $options['max_uses'] ?? null,
                'coupon_code' => $options['prefix'] ?? strtoupper(substr(preg_replace('/[^a-zA-Z0-9]/', '', $seasonalType->name), 0, 5)) . '_' . Coupon::generateUniqueCode(4),
                'is_active' => true,
            ];

            $coupons->push($this->generateCoupon($couponData));
        }

        return $coupons;
    }

    /**
     * Generate a first-order coupon
     *
     * @param string $discountType
     * @param float $discountAmount
     * @param array $options
     * @return Coupon
     */
    public function generateFirstOrderCoupon(
        string $discountType = Coupon::DISCOUNT_PERCENTAGE,
        float $discountAmount = 15.0,
        array $options = []
    ): Coupon {
        $couponData = [
            'coupon_type' => Coupon::TYPE_FIRST_ORDER,
            'discount_type' => $discountType,
            'discount_amount' => $discountAmount,
            'start_date' => $options['start_date'] ?? now(),
            'expiry_date' => $options['expiry_date'] ?? null,
            'max_uses' => $options['max_uses'] ?? null,
            'per_user_limit' => 1, // One per user
            'is_first_order_only' => true,
            'coupon_code' => $options['prefix'] ?? 'FIRST_' . Coupon::generateUniqueCode(4),
            'is_active' => true,
        ];

        return $this->generateCoupon($couponData);
    }

    /**
     * Generate a flash sale coupon
     *
     * @param string $discountType
     * @param float $discountAmount
     * @param Carbon|null $expiryDate
     * @param array $options
     * @return Coupon
     */
    public function generateFlashCoupon(
        string $discountType = Coupon::DISCOUNT_PERCENTAGE,
        float $discountAmount = 20.0,
        ?Carbon $expiryDate = null,
        array $options = []
    ): Coupon {
        // Default to 24 hours from now if not specified
        $expiryDate = $expiryDate ?? now()->addHours(24);

        $couponData = [
            'coupon_type' => Coupon::TYPE_FLASH,
            'discount_type' => $discountType,
            'discount_amount' => $discountAmount,
            'start_date' => $options['start_date'] ?? now(),
            'expiry_date' => $expiryDate,
            'max_uses' => $options['max_uses'] ?? 100, // Limit total uses
            'per_user_limit' => $options['per_user_limit'] ?? 1, // Typically one per user
            'coupon_code' => $options['prefix'] ?? 'FLASH_' . Coupon::generateUniqueCode(4),
            'is_active' => true,
        ];

        return $this->generateCoupon($couponData);
    }

    /**
     * Generate volume discount coupons for different page thresholds
     *
     * @param array $volumeThresholds Array of pages => discount amount
     * @param string $discountType
     * @param array $options
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function generateVolumeCoupons(
        array $volumeThresholds,
        string $discountType = Coupon::DISCOUNT_PERCENTAGE,
        array $options = []
    ): \Illuminate\Database\Eloquent\Collection {
        $coupons = collect();

        foreach ($volumeThresholds as $pages => $discountAmount) {
            if ($discountAmount <= 0 || $pages <= 0) {
                continue;
            }

            $couponData = [
                'coupon_type' => Coupon::TYPE_VOLUME,
                'discount_type' => $discountType,
                'discount_amount' => $discountAmount,
                'min_pages' => $pages,
                'start_date' => $options['start_date'] ?? now(),
                'expiry_date' => $options['expiry_date'] ?? null,
                'max_uses' => $options['max_uses'] ?? null,
                'coupon_code' => $options['prefix'] ?? 'VOL' . $pages . '_' . Coupon::generateUniqueCode(4),
                'is_active' => true,
            ];

            $coupons->push($this->generateCoupon($couponData));
        }

        // Convert Support Collection to Eloquent Collection before returning
        return new \Illuminate\Database\Eloquent\Collection($coupons->all());
    }


    /**
     * Generate a referral coupon
     *
     * @param int|null $referrerId
     * @param string $discountType
     * @param float $discountAmount
     * @param array $options
     * @return Coupon
     */
    public function generateReferralCoupon(
        ?int $referrerId = null,
        string $discountType = Coupon::DISCOUNT_PERCENTAGE,
        float $discountAmount = 15.0,
        array $options = []
    ): Coupon {
        $couponData = [
            'coupon_type' => Coupon::TYPE_REFERRAL,
            'discount_type' => $discountType,
            'discount_amount' => $discountAmount,
            'start_date' => $options['start_date'] ?? now(),
            'expiry_date' => $options['expiry_date'] ?? now()->addMonths(3), // Valid for 3 months
            'max_uses' => $options['max_uses'] ?? 10, // Allow multiple uses
            'per_user_limit' => $options['per_user_limit'] ?? 1, // One per referred user
            'is_referral' => true,
            'referrer_user_id' => $referrerId,
            'user_specific' => $referrerId ? true : false,
            'coupon_code' => $options['prefix'] ?? 'REF_' . Coupon::generateUniqueCode(6),
            'is_active' => true,
        ];

        return $this->generateCoupon($couponData);
    }

    /**
     * Find optimal coupon for an order
     *
     * @param User $user
     * @param array $orderContext
     * @return Coupon|null The best coupon to apply
     */
    public function findOptimalCoupon(User $user, array $orderContext): ?Coupon
    {
        $applicableCoupons = $this->getApplicableCoupons($user, $orderContext);

        if ($applicableCoupons->isEmpty()) {
            return null;
        }

        // Calculate discount amount for each coupon
        $couponDiscounts = $applicableCoupons->map(function ($coupon) use ($orderContext) {
            $discount = $this->calculateDiscount($coupon, $orderContext['order_total'] ?? 0, $orderContext['pages'] ?? 1);
            return [
                'coupon' => $coupon,
                'discount' => $discount
            ];
        });

        // Sort by discount amount descending and return the highest
        $bestCoupon = $couponDiscounts->sortByDesc('discount')->first();
        return $bestCoupon ? $bestCoupon['coupon'] : null;
    }

    /**
     * Get the best discount for a user, prioritizing special discounts over default
     *
     * @param User $user
     * @param array $orderContext
     * @return array|null Array with coupon and discount info
     */
    public function getBestDiscount(User $user, array $orderContext): ?array
    {
        $applicableCoupons = $this->getApplicableCoupons($user, $orderContext);

        if ($applicableCoupons->isEmpty()) {
            return null;
        }

        // Separate special coupons from default coupons
        $specialCoupons = $applicableCoupons->filter(function ($coupon) {
            return $coupon->coupon_type !== Coupon::TYPE_DEFAULT;
        });

        $defaultCoupons = $applicableCoupons->filter(function ($coupon) {
            return $coupon->coupon_type === Coupon::TYPE_DEFAULT;
        });

        // If user has special coupons, return the best one
        if ($specialCoupons->isNotEmpty()) {
            // Calculate discount amount for each special coupon
            $specialCouponDiscounts = $specialCoupons->map(function ($coupon) use ($orderContext) {
                $discount = $this->calculateDiscount($coupon, $orderContext['order_total'] ?? 0, $orderContext['pages'] ?? 1);
                return [
                    'coupon' => $coupon,
                    'discount' => $discount
                ];
            });

            // Sort by discount amount descending and return the highest
            $bestSpecialCoupon = $specialCouponDiscounts->sortByDesc('discount')->first();
            if ($bestSpecialCoupon) {
                return [
                    'coupon' => $bestSpecialCoupon['coupon'],
                    'discount' => $bestSpecialCoupon['discount'],
                    'type' => 'special'
                ];
            }
        }

        // If no special coupons, return the best default coupon
        if ($defaultCoupons->isNotEmpty()) {
            $bestDefaultCoupon = $defaultCoupons->sortByDesc('discount_amount')->first();
            $discount = $this->calculateDiscount($bestDefaultCoupon, $orderContext['order_total'] ?? 0, $orderContext['pages'] ?? 1);
            return [
                'coupon' => $bestDefaultCoupon,
                'discount' => $discount,
                'type' => 'default'
            ];
        }

        return null;
    }

    /**
     * Check if user is eligible for first order discount
     *
     * @param User $user
     * @return bool
     */
    public function isEligibleForFirstOrderDiscount(User $user): bool
    {
        // User is eligible if they have no completed orders
        return $user->orders()->count() === 0;
    }

    /**
     * Check if today is user's anniversary
     *
     * @param User $user
     * @return bool
     */
    public function isUserAnniversary(User $user): bool
    {
        if (!$user->created_at) {
            return false;
        }

        $today = Carbon::today();
        $registrationDate = Carbon::parse($user->created_at);

        // Anniversary if same month and day, at least one year ago
        return $today->month === $registrationDate->month &&
            $today->day === $registrationDate->day &&
            $today->year > $registrationDate->year;
    }

    /**
     * Get user's current loyalty tier
     *
     * @param User $user
     * @return LoyaltyTier|null
     */
    public function getUserLoyaltyTier(User $user): ?LoyaltyTier
    {
        $orderCount = $user->orders()->where('payment_status', 'paid')->count();
        return LoyaltyTier::getTierForOrderCount($orderCount);
    }
}
