<?php

namespace App\Services;

use App\Models\WriterInfraction;
use App\Models\WriterFine;
use App\Models\Order;
use App\Models\AdminSetting;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class FineCalculationService
{
    /**
     * Calculate fine amount based on infraction type and severity
     */
    public function calculateFineAmount(WriterInfraction $infraction): float
    {
        $baseAmount = $this->getBaseFineAmount($infraction->infraction_type, $infraction->severity_level);
        $multiplier = $this->getRepeatOffenseMultiplier($infraction->repeat_count);
        $orderValueMultiplier = $this->getOrderValueMultiplier($infraction);
        
        $finalAmount = $baseAmount * $multiplier * $orderValueMultiplier;
        
        Log::info('Fine calculation', [
            'infraction_id' => $infraction->id,
            'type' => $infraction->infraction_type,
            'severity' => $infraction->severity_level,
            'repeat_count' => $infraction->repeat_count,
            'base_amount' => $baseAmount,
            'multiplier' => $multiplier,
            'order_value_multiplier' => $orderValueMultiplier,
            'final_amount' => $finalAmount
        ]);
        
        return round($finalAmount, 2);
    }

    /**
     * Get base fine amount from configuration
     */
    private function getBaseFineAmount(string $infractionType, string $severityLevel): float
    {
        $settingKey = "fine_{$infractionType}_{$severityLevel}";
        $defaultAmount = $this->getDefaultFineAmount($infractionType, $severityLevel);
        
        return (float) AdminSetting::where('key', $settingKey)->value('value') ?? $defaultAmount;
    }

    /**
     * Get default fine amounts if not configured
     */
    private function getDefaultFineAmount(string $infractionType, string $severityLevel): float
    {
        $defaults = [
            'deadline_violation' => [
                'warning' => 5.00,
                'minor' => 15.00,
                'major' => 35.00,
                'critical' => 75.00
            ],
            'quality_issue' => [
                'warning' => 10.00,
                'minor' => 25.00,
                'major' => 50.00,
                'critical' => 100.00
            ],
            'communication_failure' => [
                'warning' => 5.00,
                'minor' => 15.00,
                'major' => 30.00,
                'critical' => 60.00
            ],
            'policy_violation' => [
                'warning' => 15.00,
                'minor' => 35.00,
                'major' => 75.00,
                'critical' => 150.00
            ],
            'plagiarism' => [
                'warning' => 25.00,
                'minor' => 50.00,
                'major' => 100.00,
                'critical' => 200.00
            ],
            'instruction_non_compliance' => [
                'warning' => 10.00,
                'minor' => 25.00,
                'major' => 50.00,
                'critical' => 100.00
            ],
            'unresponsive' => [
                'warning' => 5.00,
                'minor' => 15.00,
                'major' => 30.00,
                'critical' => 60.00
            ],
            'unprofessional_behavior' => [
                'warning' => 10.00,
                'minor' => 25.00,
                'major' => 50.00,
                'critical' => 100.00
            ]
        ];

        return $defaults[$infractionType][$severityLevel] ?? 25.00;
    }

    /**
     * Get multiplier for repeat offenses
     */
    private function getRepeatOffenseMultiplier(int $repeatCount): float
    {
        if ($repeatCount <= 1) {
            return 1.0;
        }
        
        // Progressive increase: 1.5x, 2x, 3x, 5x
        $multipliers = [1.0, 1.5, 2.0, 3.0, 5.0];
        $index = min($repeatCount - 1, count($multipliers) - 1);
        
        return $multipliers[$index];
    }

    /**
     * Get multiplier based on order value (for percentage-based fines)
     */
    private function getOrderValueMultiplier(WriterInfraction $infraction): float
    {
        if (!$infraction->order_id) {
            return 1.0;
        }

        $order = Order::find($infraction->order_id);
        if (!$order) {
            return 1.0;
        }

        // For percentage-based fines, calculate based on order value
        $percentageFines = ['quality_issue', 'plagiarism', 'deadline_violation'];
        if (in_array($infraction->infraction_type, $percentageFines)) {
            $percentage = $this->getPercentageFineRate($infraction->infraction_type, $infraction->severity_level);
            return ($order->order_amount * $percentage) / 100;
        }

        return 1.0;
    }

    /**
     * Get percentage fine rate for specific infraction types
     */
    private function getPercentageFineRate(string $infractionType, string $severityLevel): float
    {
        $rates = [
            'quality_issue' => [
                'warning' => 2.0,
                'minor' => 5.0,
                'major' => 10.0,
                'critical' => 20.0
            ],
            'plagiarism' => [
                'warning' => 5.0,
                'minor' => 10.0,
                'major' => 20.0,
                'critical' => 35.0
            ],
            'deadline_violation' => [
                'warning' => 1.0,
                'minor' => 3.0,
                'major' => 7.0,
                'critical' => 15.0
            ]
        ];

        return $rates[$infractionType][$severityLevel] ?? 5.0;
    }

    /**
     * Automatically create fine for an infraction
     */
    public function createAutomaticFine(WriterInfraction $infraction): ?WriterFine
    {
        try {
            $fineAmount = $this->calculateFineAmount($infraction);
            
            $fine = WriterFine::create([
                'writer_id' => $infraction->writer_id,
                'infraction_id' => $infraction->id,
                'order_id' => $infraction->order_id,
                'applied_by' => $infraction->reported_by,
                'fine_type' => $this->determineFineType($infraction),
                'fine_amount' => $fineAmount,
                'currency' => 'USD',
                'fine_reason' => $this->generateFineReason($infraction),
                'calculation_details' => $this->getCalculationDetails($infraction, $fineAmount),
                'application_method' => $this->determineApplicationMethod($infraction),
                'status' => 'pending',
                'applied_at' => now(),
                'effective_from' => $this->determineEffectiveDate($infraction),
                'is_automatic' => true,
            ]);

            Log::info('Automatic fine created', [
                'fine_id' => $fine->id,
                'infraction_id' => $infraction->id,
                'amount' => $fineAmount
            ]);

            return $fine;
        } catch (\Exception $e) {
            Log::error('Failed to create automatic fine', [
                'infraction_id' => $infraction->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Determine the type of fine to apply
     */
    private function determineFineType(WriterInfraction $infraction): string
    {
        // Critical infractions and repeat offenses get progressive fines
        if ($infraction->severity_level === 'critical' || $infraction->repeat_count > 2) {
            return 'progressive';
        }

        // Quality issues and plagiarism get percentage-based fines
        if (in_array($infraction->infraction_type, ['quality_issue', 'plagiarism'])) {
            return 'percentage_based';
        }

        // Deadline violations get opportunity cost fines
        if ($infraction->infraction_type === 'deadline_violation') {
            return 'opportunity_cost';
        }

        // Default to fixed amount
        return 'fixed_amount';
    }

    /**
     * Generate human-readable fine reason
     */
    private function generateFineReason(WriterInfraction $infraction): string
    {
        $severity = ucfirst($infraction->severity_level);
        $type = str_replace('_', ' ', $infraction->infraction_type);
        
        if ($infraction->repeat_count > 1) {
            return "{$severity} {$type} - Repeat offense (#{$infraction->repeat_count})";
        }
        
        return "{$severity} {$type}";
    }

    /**
     * Get detailed calculation breakdown
     */
    private function getCalculationDetails(WriterInfraction $infraction, float $finalAmount): array
    {
        $baseAmount = $this->getBaseFineAmount($infraction->infraction_type, $infraction->severity_level);
        $multiplier = $this->getRepeatOffenseMultiplier($infraction->repeat_count);
        
        return [
            'base_amount' => $baseAmount,
            'repeat_multiplier' => $multiplier,
            'repeat_count' => $infraction->repeat_count,
            'order_value_multiplier' => $this->getOrderValueMultiplier($infraction),
            'calculation_method' => $this->determineFineType($infraction),
            'final_amount' => $finalAmount
        ];
    }

    /**
     * Determine when the fine should take effect
     */
    private function determineEffectiveDate(WriterInfraction $infraction): ?Carbon
    {
        // Critical infractions take effect immediately
        if ($infraction->severity_level === 'critical') {
            return null; // null means immediately effective
        }

        // Major infractions take effect after 24 hours
        if ($infraction->severity_level === 'major') {
            return now()->addHours(24);
        }

        // Minor infractions take effect after 48 hours
        if ($infraction->severity_level === 'minor') {
            return now()->addHours(48);
        }

        // Warnings take effect after 72 hours
        return now()->addHours(72);
    }

    /**
     * Determine how the fine should be applied
     */
    private function determineApplicationMethod(WriterInfraction $infraction): string
    {
        // Critical infractions are deducted immediately
        if ($infraction->severity_level === 'critical') {
            return 'immediate';
        }

        // Repeat offenses are deducted immediately
        if ($infraction->repeat_count > 2) {
            return 'immediate';
        }

        // Default to future deduction
        return 'future_deduction';
    }

    /**
     * Check if an infraction should trigger automatic fine
     */
    public function shouldTriggerAutomaticFine(WriterInfraction $infraction): bool
    {
        // All confirmed infractions trigger fines
        if ($infraction->status === 'confirmed') {
            return true;
        }

        // Critical infractions trigger fines immediately
        if ($infraction->severity_level === 'critical') {
            return true;
        }

        // Repeat offenses trigger fines
        if ($infraction->repeat_count > 1) {
            return true;
        }

        return false;
    }

    /**
     * Get fine statistics for a writer
     */
    public function getWriterFineStatistics(int $writerId): array
    {
        $totalFines = WriterFine::where('writer_id', $writerId)->count();
        $activeFines = WriterFine::where('writer_id', $writerId)
            ->whereIn('status', ['pending', 'applied'])
            ->count();
        $totalAmount = WriterFine::where('writer_id', $writerId)->sum('fine_amount');
        $paidAmount = WriterFine::where('writer_id', $writerId)
            ->where('status', 'applied')
            ->sum('amount_deducted');

        return [
            'total_fines' => $totalFines,
            'active_fines' => $activeFines,
            'total_amount' => $totalAmount,
            'paid_amount' => $paidAmount,
            'remaining_amount' => $totalAmount - $paidAmount
        ];
    }
} 