<?php

namespace App\Services;

use App\Models\PageMedia;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Facades\Image;

class MediaService
{
    /**
     * Upload and process media file
     */
    public function uploadMedia(UploadedFile $file, int $pageId, array $metadata = []): PageMedia
    {
        // Generate unique filename
        $filename = $this->generateUniqueFilename($file);
        $filePath = $this->getUploadPath($pageId) . '/' . $filename;

        // Store the file
        $storedPath = $file->storeAs($this->getUploadPath($pageId), $filename, 'public');

        // Get file info
        $fileInfo = [
            'file_path' => 'storage/' . $storedPath,
            'file_size' => $file->getSize(),
            'mime_type' => $file->getMimeType(),
            'original_name' => $file->getClientOriginalName(),
        ];

        // Process image if it's an image
        if ($this->isImage($file)) {
            $this->processImage($fileInfo['file_path']);
            $fileInfo['is_optimized'] = true;
        }

        // Create media record
        return PageMedia::create([
            'page_id' => $pageId,
            'media_type' => $this->getMediaType($file),
            'file_path' => $fileInfo['file_path'],
            'alt_text' => $metadata['alt_text'] ?? '',
            'title' => $metadata['title'] ?? $file->getClientOriginalName(),
            'caption' => $metadata['caption'] ?? '',
            'usage_context' => $metadata['usage_context'] ?? '',
            'file_size' => $fileInfo['file_size'],
            'mime_type' => $fileInfo['mime_type'],
            'is_optimized' => $fileInfo['is_optimized'] ?? false,
            'optimization_settings' => $this->getOptimizationSettings($file),
        ]);
    }

    /**
     * Update media metadata
     */
    public function updateMediaMetadata(PageMedia $media, array $metadata): PageMedia
    {
        $media->update($metadata);
        return $media->fresh();
    }

    /**
     * Delete media file and record
     */
    public function deleteMedia(PageMedia $media): bool
    {
        // Delete physical file
        $this->deletePhysicalFile($media->file_path);

        // Delete media record
        return $media->delete();
    }

    /**
     * Get media files for a page
     */
    public function getPageMedia(int $pageId, string $context = null): \Illuminate\Database\Eloquent\Collection
    {
        $query = PageMedia::where('page_id', $pageId);

        if ($context) {
            $query->where('usage_context', $context);
        }

        return $query->orderBy('created_at', 'desc')->get();
    }

    /**
     * Get media by type
     */
    public function getMediaByType(int $pageId, string $mediaType): \Illuminate\Database\Eloquent\Collection
    {
        return PageMedia::where('page_id', $pageId)
            ->where('media_type', $mediaType)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Generate optimized image sizes
     */
    public function generateImageSizes(string $imagePath, array $sizes = null): array
    {
        if (!$sizes) {
            $sizes = [
                'thumbnail' => [150, 150],
                'medium' => [400, 400],
                'large' => [800, 600],
                'xlarge' => [1200, 900],
            ];
        }

        $generatedSizes = [];
        $originalPath = public_path($imagePath);

        if (!file_exists($originalPath)) {
            return $generatedSizes;
        }

        foreach ($sizes as $sizeName => $dimensions) {
            $width = $dimensions[0];
            $height = $dimensions[1];

            // Generate new filename
            $pathInfo = pathinfo($originalPath);
            $newFilename = $pathInfo['filename'] . '_' . $sizeName . '.' . $pathInfo['extension'];
            $newPath = $pathInfo['dirname'] . '/' . $newFilename;

            try {
                // Resize image
                Image::make($originalPath)
                    ->resize($width, $height, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    })
                    ->save($newPath, 85); // 85% quality

                $generatedSizes[$sizeName] = [
                    'path' => str_replace(public_path() . '/', '', $newPath),
                    'width' => $width,
                    'height' => $height,
                    'size' => filesize($newPath)
                ];
            } catch (\Exception $e) {
                // Log error but continue
                \Log::error('Image resize failed: ' . $e->getMessage());
            }
        }

        return $generatedSizes;
    }

    /**
     * Optimize existing image
     */
    public function optimizeImage(string $imagePath, int $quality = 85): bool
    {
        $fullPath = public_path($imagePath);

        if (!file_exists($fullPath) || !$this->isImageFile($fullPath)) {
            return false;
        }

        try {
            Image::make($fullPath)
                ->save($fullPath, $quality);

            return true;
        } catch (\Exception $e) {
            \Log::error('Image optimization failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get media usage statistics
     */
    public function getMediaStats(int $pageId = null): array
    {
        $query = PageMedia::query();

        if ($pageId) {
            $query->where('page_id', $pageId);
        }

        $media = $query->get();

        return [
            'total_files' => $media->count(),
            'total_size' => $media->sum('file_size'),
            'by_type' => $media->groupBy('media_type')->map(function ($group) {
                return [
                    'count' => $group->count(),
                    'total_size' => $group->sum('file_size')
                ];
            }),
            'by_context' => $media->groupBy('usage_context')->map(function ($group) {
                return [
                    'count' => $group->count(),
                    'total_size' => $group->sum('file_size')
                ];
            }),
            'optimized_count' => $media->where('is_optimized', true)->count(),
            'unoptimized_count' => $media->where('is_optimized', false)->count(),
        ];
    }

    /**
     * Search media files
     */
    public function searchMedia(string $searchTerm, int $pageId = null): \Illuminate\Database\Eloquent\Collection
    {
        $query = PageMedia::query();

        if ($pageId) {
            $query->where('page_id', $pageId);
        }

        return $query->where(function ($q) use ($searchTerm) {
            $q->where('title', 'like', '%' . $searchTerm . '%')
              ->orWhere('alt_text', 'like', '%' . $searchTerm . '%')
              ->orWhere('caption', 'like', '%' . $searchTerm . '%')
              ->orWhere('usage_context', 'like', '%' . $searchTerm . '%');
        })
        ->orderBy('created_at', 'desc')
        ->get();
    }

    /**
     * Bulk optimize images
     */
    public function bulkOptimizeImages(int $pageId = null, int $quality = 85): array
    {
        $query = PageMedia::where('media_type', 'image');

        if ($pageId) {
            $query->where('page_id', $pageId);
        }

        $images = $query->get();
        $results = ['success' => 0, 'failed' => 0, 'errors' => []];

        foreach ($images as $image) {
            if ($this->optimizeImage($image->file_path, $quality)) {
                $image->update(['is_optimized' => true]);
                $results['success']++;
            } else {
                $results['failed']++;
                $results['errors'][] = 'Failed to optimize: ' . $image->file_path;
            }
        }

        return $results;
    }

    /**
     * Generate unique filename
     */
    private function generateUniqueFilename(UploadedFile $file): string
    {
        $extension = $file->getClientOriginalExtension();
        $filename = Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
        $uniqueId = time() . '_' . Str::random(8);
        
        return $filename . '_' . $uniqueId . '.' . $extension;
    }

    /**
     * Get upload path for page
     */
    private function getUploadPath(int $pageId): string
    {
        return 'pages/' . $pageId;
    }

    /**
     * Check if file is an image
     */
    private function isImage(UploadedFile $file): bool
    {
        return str_starts_with($file->getMimeType(), 'image/');
    }

    /**
     * Check if file path is an image
     */
    private function isImageFile(string $filePath): bool
    {
        $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));
        
        return in_array($extension, $imageExtensions);
    }

    /**
     * Process uploaded image
     */
    private function processImage(string $imagePath): void
    {
        $fullPath = public_path($imagePath);

        if (!file_exists($fullPath)) {
            return;
        }

        try {
            // Optimize the original image
            Image::make($fullPath)
                ->save($fullPath, 85);

            // Generate different sizes
            $this->generateImageSizes($imagePath);
        } catch (\Exception $e) {
            \Log::error('Image processing failed: ' . $e->getMessage());
        }
    }

    /**
     * Get media type from file
     */
    private function getMediaType(UploadedFile $file): string
    {
        $mimeType = $file->getMimeType();

        if (str_starts_with($mimeType, 'image/')) {
            return 'image';
        } elseif (str_starts_with($mimeType, 'video/')) {
            return 'video';
        } else {
            return 'document';
        }
    }

    /**
     * Get optimization settings for file
     */
    private function getOptimizationSettings(UploadedFile $file): array
    {
        $settings = [
            'uploaded_at' => now()->toISOString(),
            'original_size' => $file->getSize(),
            'original_name' => $file->getClientOriginalName(),
        ];

        if ($this->isImage($file)) {
            $settings['image_processing'] = [
                'quality' => 85,
                'resize' => true,
                'generate_sizes' => true,
            ];
        }

        return $settings;
    }

    /**
     * Delete physical file from storage
     */
    private function deletePhysicalFile(string $filePath): void
    {
        $fullPath = public_path($filePath);

        if (file_exists($fullPath)) {
            unlink($fullPath);
        }

        // Also delete any generated sizes
        $pathInfo = pathinfo($fullPath);
        $directory = $pathInfo['dirname'];
        $filename = $pathInfo['filename'];
        $extension = $pathInfo['extension'];

        $sizes = ['thumbnail', 'medium', 'large', 'xlarge'];
        
        foreach ($sizes as $size) {
            $sizePath = $directory . '/' . $filename . '_' . $size . '.' . $extension;
            if (file_exists($sizePath)) {
                unlink($sizePath);
            }
        }
    }

    /**
     * Validate uploaded file
     */
    public function validateFile(UploadedFile $file): array
    {
        $errors = [];

        // Check file size (10MB limit)
        if ($file->getSize() > 10 * 1024 * 1024) {
            $errors[] = 'File size must be less than 10MB';
        }

        // Check file type
        $allowedMimes = [
            'image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml',
            'video/mp4', 'video/avi', 'video/mov',
            'application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];

        if (!in_array($file->getMimeType(), $allowedMimes)) {
            $errors[] = 'File type not allowed';
        }

        // Check image dimensions if it's an image
        if ($this->isImage($file)) {
            $imageInfo = getimagesize($file->getPathname());
            if ($imageInfo && ($imageInfo[0] > 4000 || $imageInfo[1] > 4000)) {
                $errors[] = 'Image dimensions too large (max 4000x4000)';
            }
        }

        return $errors;
    }

    /**
     * Get file URL
     */
    public function getFileUrl(string $filePath): string
    {
        return asset($filePath);
    }

    /**
     * Get optimized image URL
     */
    public function getOptimizedImageUrl(string $imagePath, string $size = 'medium'): string
    {
        $pathInfo = pathinfo($imagePath);
        $optimizedPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '_' . $size . '.' . $pathInfo['extension'];

        // Check if optimized version exists
        if (file_exists(public_path($optimizedPath))) {
            return asset($optimizedPath);
        }

        // Fallback to original
        return asset($imagePath);
    }
}
