<?php

namespace App\Services;

use App\Models\Page;
use App\Models\PageContent;
use App\Models\PageSeo;
use App\Models\PageMedia;
use App\Models\PageRelationship;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class PageService
{
    /**
     * Create a new page with all related data
     */
    public function createPage(array $pageData, array $contentBlocks = [], array $seoData = [], array $mediaFiles = []): Page
    {
        return DB::transaction(function () use ($pageData, $contentBlocks, $seoData, $mediaFiles) {
            // Generate slug if not provided
            if (empty($pageData['slug'])) {
                $pageData['slug'] = Str::slug($pageData['title']);
            }

            // Ensure slug is unique
            $pageData['slug'] = $this->ensureUniqueSlug($pageData['slug']);

            // Create the main page
            $page = Page::create($pageData);

            // Create content blocks
            if (!empty($contentBlocks)) {
                $this->createContentBlocks($page->id, $contentBlocks);
            }

            // Create SEO data
            if (!empty($seoData)) {
                $this->createSeoData($page->id, $seoData);
            }

            // Create media files
            if (!empty($mediaFiles)) {
                $this->createMediaFiles($page->id, $mediaFiles);
            }

            return $page->load(['contentBlocks', 'seo', 'media', 'relationships']);
        });
    }

    /**
     * Update an existing page with all related data
     */
    public function updatePage(Page $page, array $pageData, array $contentBlocks = [], array $seoData = [], array $mediaFiles = []): Page
    {
        return DB::transaction(function () use ($page, $pageData, $contentBlocks, $seoData, $mediaFiles) {
            $oldSlug = $page->slug;
            
            // Update slug if title changed
            if (isset($pageData['title']) && $pageData['title'] !== $page->title) {
                $pageData['slug'] = $this->ensureUniqueSlug(
                    Str::slug($pageData['title']), 
                    $page->id
                );
            }

            // Update the main page
            $page->update($pageData);

            // Update content blocks
            if (!empty($contentBlocks)) {
                $this->updateContentBlocks($page->id, $contentBlocks);
            }

            // Update SEO data
            if (!empty($seoData)) {
                $this->updateSeoData($page->id, $seoData);
            }

            // Update media files
            if (!empty($mediaFiles)) {
                $this->updateMediaFiles($page->id, $mediaFiles);
            }

            // Clear cache for this page
            $this->clearPageCache($oldSlug);
            if ($page->slug !== $oldSlug) {
                $this->clearPageCache($page->slug);
            }

            return $page->fresh(['contentBlocks', 'seo', 'media', 'relationships']);
        });
    }

    /**
     * Get page with all related data (with caching)
     */
    public function getPageWithContent(string $slug): ?Page
    {
        // Cache key for this specific page
        $cacheKey = "page_content_{$slug}";
        
        // Cache for 1 hour (3600 seconds)
        return Cache::remember($cacheKey, 3600, function () use ($slug) {
            return Page::where('slug', $slug)
                ->where('status', 'published')
                ->with(['contentBlocks' => function ($query) {
                    $query->where('is_active', true)->orderBy('order');
                }, 'seo'])
                ->first();
        });
    }

    /**
     * Clear cache for a specific page
     */
    public function clearPageCache(string $slug): void
    {
        Cache::forget("page_content_{$slug}");
    }

    /**
     * Clear all page caches
     */
    public function clearAllPageCaches(): void
    {
        // Get all page slugs and clear their caches
        $slugs = Page::pluck('slug');
        foreach ($slugs as $slug) {
            $this->clearPageCache($slug);
        }
    }

    /**
     * Get all pages for admin listing
     */
    public function getAllPages(array $filters = [])
    {
        $query = Page::with(['seo', 'parent', 'children']);

        // Apply filters
        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['page_type'])) {
            $query->where('page_type', $filters['page_type']);
        }

        if (isset($filters['search'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('title', 'like', '%' . $filters['search'] . '%')
                  ->orWhere('slug', 'like', '%' . $filters['search'] . '%');
            });
        }

        return $query->orderBy('created_at', 'desc')->paginate(5);
    }

    /**
     * Delete page and all related data
     */
    public function deletePage(Page $page): bool
    {
        return DB::transaction(function () use ($page) {
            // Delete media files from storage
            foreach ($page->media as $media) {
                $this->deleteMediaFile($media->file_path);
            }

            // Delete the page (cascade will handle related records)
            return $page->delete();
        });
    }

    /**
     * Create content blocks for a page
     */
    private function createContentBlocks(int $pageId, array $contentBlocks): void
    {
        foreach ($contentBlocks as $index => $block) {
            PageContent::create([
                'page_id' => $pageId,
                'block_type' => $block['block_type'],
                'block_key' => $block['block_key'] ?? null,
                'content' => $block['content'],
                'order' => $block['order'] ?? $index,
                'is_active' => $block['is_active'] ?? true,
            ]);
        }
    }

    /**
     * Update content blocks for a page
     */
    private function updateContentBlocks(int $pageId, array $contentBlocks): void
    {
        \Log::info('PageService updateContentBlocks', [
            'page_id' => $pageId,
            'blocks_count' => count($contentBlocks),
            'blocks' => array_map(function($block) {
                return [
                    'type' => $block['block_type'] ?? 'N/A',
                    'key' => $block['block_key'] ?? 'N/A',
                    'content_size' => !empty($block['content']) ? count($block['content']) : 0
                ];
            }, $contentBlocks)
        ]);

        // Delete existing content blocks
        $deleted = PageContent::where('page_id', $pageId)->delete();
        \Log::info('Deleted existing blocks', ['count' => $deleted]);

        // Create new content blocks
        $this->createContentBlocks($pageId, $contentBlocks);
        
        $newCount = PageContent::where('page_id', $pageId)->count();
        \Log::info('Created new blocks', ['count' => $newCount]);
    }

    /**
     * Create SEO data for a page
     */
    private function createSeoData(int $pageId, array $seoData): void
    {
        PageSeo::create([
            'page_id' => $pageId,
            ...$seoData
        ]);
    }

    /**
     * Update SEO data for a page
     */
    private function updateSeoData(int $pageId, array $seoData): void
    {
        // Sanitize structured_data if it's a string (should be array or null)
        if (isset($seoData['structured_data']) && is_string($seoData['structured_data'])) {
            \Log::warning('structured_data received as string, attempting to parse', [
                'page_id' => $pageId,
                'string_value' => $seoData['structured_data']
            ]);
            
            try {
                $seoData['structured_data'] = json_decode($seoData['structured_data'], true);
            } catch (\Exception $e) {
                \Log::error('Failed to parse structured_data string', [
                    'page_id' => $pageId,
                    'error' => $e->getMessage()
                ]);
                // Set to null if parsing fails
                $seoData['structured_data'] = null;
            }
        }
        
        // Log the SEO data being updated
        \Log::info('PageService updateSeoData called', [
            'page_id' => $pageId,
            'seo_fields' => array_keys($seoData),
            'has_structured_data' => isset($seoData['structured_data']),
            'structured_data_type' => isset($seoData['structured_data']) ? gettype($seoData['structured_data']) : 'not set',
            'structured_data_value' => isset($seoData['structured_data']) ? $seoData['structured_data'] : null,
            'has_meta_title' => isset($seoData['meta_title']),
            'meta_title_value' => $seoData['meta_title'] ?? null,
            'has_meta_description' => isset($seoData['meta_description']),
            'meta_description_value' => $seoData['meta_description'] ?? null,
        ]);
        
        $result = PageSeo::updateOrCreate(
            ['page_id' => $pageId],
            $seoData
        );
        
        \Log::info('PageService SEO data saved to database', [
            'page_id' => $pageId,
            'seo_id' => $result->id,
            'saved_structured_data' => $result->structured_data,
            'saved_meta_title' => $result->meta_title,
            'saved_meta_description' => $result->meta_description,
        ]);
    }

    /**
     * Create media files for a page
     */
    private function createMediaFiles(int $pageId, array $mediaFiles): void
    {
        foreach ($mediaFiles as $media) {
            PageMedia::create([
                'page_id' => $pageId,
                ...$media
            ]);
        }
    }

    /**
     * Update media files for a page
     */
    private function updateMediaFiles(int $pageId, array $mediaFiles): void
    {
        // Delete existing media files
        $existingMedia = PageMedia::where('page_id', $pageId)->get();
        foreach ($existingMedia as $media) {
            $this->deleteMediaFile($media->file_path);
        }
        PageMedia::where('page_id', $pageId)->delete();

        // Create new media files
        $this->createMediaFiles($pageId, $mediaFiles);
    }

    /**
     * Ensure slug is unique
     */
    private function ensureUniqueSlug(string $slug, ?int $excludeId = null): string
    {
        $originalSlug = $slug;
        $counter = 1;

        while (Page::where('slug', $slug)
            ->when($excludeId, function ($query, $excludeId) {
                return $query->where('id', '!=', $excludeId);
            })
            ->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    /**
     * Delete media file from storage
     */
    private function deleteMediaFile(string $filePath): void
    {
        if (file_exists(public_path($filePath))) {
            unlink(public_path($filePath));
        }
    }

    /**
     * Duplicate a page
     */
    public function duplicatePage(Page $page, string $newTitle): Page
    {
        return DB::transaction(function () use ($page, $newTitle) {
            // Create new page data
            $newPageData = $page->toArray();
            unset($newPageData['id'], $newPageData['created_at'], $newPageData['updated_at']);
            $newPageData['title'] = $newTitle;
            $newPageData['status'] = 'draft';

            // Create content blocks data
            $contentBlocks = $page->contentBlocks->map(function ($block) {
                return [
                    'block_type' => $block->block_type,
                    'block_key' => $block->block_key,
                    'content' => $block->content,
                    'order' => $block->order,
                    'is_active' => $block->is_active,
                ];
            })->toArray();

            // Create SEO data
            $seoData = $page->seo ? $page->seo->toArray() : [];
            unset($seoData['id'], $seoData['page_id'], $seoData['created_at'], $seoData['updated_at']);

            // Create new page
            return $this->createPage($newPageData, $contentBlocks, $seoData);
        });
    }
}
