<?php

namespace App\Services\Payments;

use App\Models\Order;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Services\OrderStatusService;
use Illuminate\Support\Facades\Log;

abstract class AbstractPaymentService implements PaymentServiceInterface
{
    /**
     * @var PaymentMethod
     */
    protected $paymentMethod;
    protected OrderStatusService $orderStatusService;

    /**
     * Create a new payment service instance.
     *
     * @param PaymentMethod $paymentMethod
     */
    public function __construct(PaymentMethod $paymentMethod, OrderStatusService $orderStatusService)
    {
        $this->paymentMethod = $paymentMethod;
        $this->orderStatusService = $orderStatusService;
    }

    /**
     * Create a new payment record for an order.
     *
     * @param Order $order
     * @param string $status
     * @param string|null $transactionId
     * @param array $metadata
     * @return Payment
     */
    protected function createPaymentRecord(
        Order $order,
        string $status = Payment::STATUS_PENDING,
        ?string $transactionId = null,
        array $metadata = []
    ): Payment {
        $payment = new Payment([
            'order_id' => $order->id,
            'payment_method' => $this->paymentMethod->name,
            'amount' => $order->net_amount,
            'currency' => $order->currency ?? 'USD',
            'status' => $status,
            'transaction_id' => $transactionId,
            'payment_date' => now(),
            'metadata' => $metadata,
        ]);

        $payment->save();

        // Log payment creation
        Log::info('Payment record created', [
            'order_id' => $order->id,
            'payment_id' => $payment->id,
            'status' => $status,
            'method' => $this->paymentMethod->name,
            'currency' => $payment->currency
        ]);

        return $payment;
    }

    /**
     * Update an existing payment record.
     *
     * @param Payment $payment
     * @param string $status
     * @param string|null $transactionId
     * @param array $metadata
     * @return Payment
     */
    protected function updatePaymentRecord(
        Payment $payment,
        string $status,
        ?string $transactionId = null,
        array $metadata = []
    ): Payment {
        $payment->status = $status;

        if ($transactionId) {
            $payment->transaction_id = $transactionId;
        }

        if (!empty($metadata)) {
            $payment->metadata = array_merge($payment->metadata ?? [], $metadata);
        }

        $payment->save();

        // Update order payment status if payment is completed
        if ($status === Payment::STATUS_COMPLETED) {
            $order = $payment->order;
            $order->payment_status = 'paid';
            $order->save();

            // Use OrderStatusService to handle status transition
            try {
                $this->orderStatusService->handlePaymentCompleted($order);
                
                Log::info('Order status updated after payment completion', [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'old_status' => 'pending',
                    'new_status' => $order->order_status,
                    'payment_id' => $payment->id
                ]);
            } catch (\Exception $e) {
                Log::error('Failed to update order status after payment completion', [
                    'order_id' => $order->id,
                    'payment_id' => $payment->id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        // Log payment update
        Log::info('Payment record updated', [
            'payment_id' => $payment->id,
            'order_id' => $payment->order_id,
            'status' => $status,
            'transaction_id' => $transactionId
        ]);

        return $payment;
    }

    /**
     * Handle errors in payment processing.
     *
     * @param Order $order
     * @param Payment|null $payment
     * @param \Exception $exception
     * @return Payment
     */
    protected function handlePaymentError(Order $order, ?Payment $payment, \Exception $exception): Payment
    {
        Log::error('Payment processing error', [
            'order_id' => $order->id,
            'payment_id' => $payment ? $payment->id : null,
            'error' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);

        if ($payment) {
            return $this->updatePaymentRecord(
                $payment,
                Payment::STATUS_FAILED,
                null,
                ['error' => $exception->getMessage()]
            );
        }

        return $this->createPaymentRecord(
            $order,
            Payment::STATUS_FAILED,
            null,
            ['error' => $exception->getMessage()]
        );
    }
}
