<?php

namespace App\Services\Payments;

use App\Models\Order;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Services\WriterNotificationService;
use Illuminate\Support\Facades\Log;
use App\Services\AdminNotificationService;
use App\Notifications\AdminPaymentNotification;
use App\Notifications\PaymentFailedNotification;
use App\Notifications\PaymentSuccessfulNotification;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use App\Services\OrderStatusService;

class PayPalDirectPaymentService extends AbstractPaymentService
{
    /**
     * Create a new PayPal Direct payment service instance.
     */
    public function __construct(PaymentMethod $paymentMethod, OrderStatusService $orderStatusService)
    {
        parent::__construct($paymentMethod, $orderStatusService);
    }

    /**
     * Initialize a payment for an order.
     * Returns form data for PayPal Standard form submission.
     *
     * @param Order $order
     * @return array Payment initialization data
     */
    public function initializePayment(Order $order): array
    {
        // Create a pending payment record
        $payment = $this->createPaymentRecord($order);

        try {
            return [
                'success' => true,
                'payment_id' => $payment->id,
                'form_data' => [
                    'action' => $this->getFormAction(),
                    'business' => $this->getBusinessEmail(),
                    'cmd' => '_xclick',
                    'item_name' => "Order #{$order->order_number}",
                    'amount' => number_format($order->net_amount, 2, '.', ''),
                    'currency_code' => $order->currency,
                    'invoice' => $order->order_number,
                    'custom' => $this->generateSecureToken($order, $payment),
                    'notify_url' => route('webhooks.paypal-direct.ipn'),
                    'return' => route('payments.paypal-direct.return', $order),
                    'cancel_return' => route('payments.paypal-direct.cancel', $order),
                    'no_shipping' => '1',
                    'rm' => '2', // Return method: POST
                    'bn' => 'AcademicScribe_SP', // Build notation
                ]
            ];
        } catch (\Exception $e) {
            $this->handlePaymentError($order, $payment, $e);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Process a payment after user confirmation.
     * This method is not used for direct form submissions but kept for interface compliance.
     *
     * @param Order $order
     * @param array $paymentData
     * @return Payment
     */
    public function processPayment(Order $order, array $paymentData): Payment
    {
        // For direct form submissions, this method is not used
        // Payment processing happens via IPN
        throw new \Exception('PayPal Direct payments are processed via IPN, not through processPayment');
    }

    /**
     * Verify a payment using provider-specific methods.
     * For PayPal Direct, this verifies the IPN data.
     *
     * @param string $transactionId
     * @return bool
     */
    public function verifyPayment(string $transactionId): bool
    {
        // Find the payment by transaction ID
        $payment = Payment::where('transaction_id', $transactionId)
            ->where('payment_method', $this->paymentMethod->name)
            ->first();

        if (!$payment) {
            Log::error('Payment not found for PayPal Direct transaction', [
                'transaction_id' => $transactionId
            ]);
            return false;
        }

        // For PayPal Direct, verification happens during IPN processing
        // This method can be used for manual verification if needed
        return $payment->status === Payment::STATUS_COMPLETED;
    }

    /**
     * Handle webhook events from the payment provider.
     * For PayPal Direct, this handles IPN notifications.
     *
     * @param array $payload
     * @return bool
     */
    public function handleWebhook(array $payload): bool
    {
        try {
            // Verify the IPN with PayPal
            $verified = $this->verifyWithPayPal($payload);
            if (!$verified) {
                Log::warning('PayPal Direct IPN verification failed', ['payload' => $payload]);
                return false;
            }

            // Process the verified IPN
            return $this->processIpnPayment($payload);
        } catch (\Exception $e) {
            Log::error('Error processing PayPal Direct IPN: ' . $e->getMessage(), [
                'payload' => $payload,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Get the client-side configuration needed for the payment method.
     *
     * @return array
     */
    public function getClientConfig(): array
    {
        return [
            'mode' => $this->paymentMethod->getConfig('mode', 'sandbox'),
            'business_email' => $this->getBusinessEmail(),
            'form_action' => $this->getFormAction(),
        ];
    }

    /**
     * Test connection with payment gateway.
     *
     * @return array
     */
    public function testConnection(): array
    {
        try {
            $businessEmail = $this->getBusinessEmail();
            $formAction = $this->getFormAction();

            if (empty($businessEmail) || empty($formAction)) {
                return [
                    'success' => false,
                    'message' => 'Missing required PayPal Direct configuration',
                ];
            }

            return [
                'success' => true,
                'message' => 'PayPal Direct configuration is valid',
                'details' => [
                    'environment' => $this->paymentMethod->getConfig('mode', 'sandbox') === 'sandbox' ? 'Sandbox' : 'Production',
                    'business_email' => substr($businessEmail, 0, 3) . '***' . substr($businessEmail, -10),
                    'form_action' => $formAction,
                ],
            ];
        } catch (\Exception $e) {
            Log::error('PayPal Direct connection test failed', [
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'PayPal Direct connection test failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Process a refund for a payment.
     *
     * @param Payment $payment
     * @param float $amount
     * @param string $reason
     * @return array
     */
    public function processRefund(Payment $payment, float $amount, string $reason): array
    {
        // PayPal Direct doesn't support automatic refunds via API
        // Refunds would need to be processed manually through PayPal dashboard
        return [
            'success' => false,
            'message' => 'Automatic refunds are not supported for PayPal Direct payments. Please process refunds manually through PayPal dashboard.',
        ];
    }

    /**
     * Verify IPN data with PayPal.
     *
     * @param array $ipnData
     * @return bool
     */
    protected function verifyWithPayPal(array $ipnData): bool
    {
        try {
            // Add the verification command
            $verifyData = $ipnData;
            $verifyData['cmd'] = '_notify-validate';

            // Get the verification URL based on mode
            $verifyUrl = $this->getIpnVerifyUrl();

            // Send verification request to PayPal
            $response = Http::asForm()->post($verifyUrl, $verifyData);

            if ($response->successful()) {
                $responseBody = trim($response->body());
                Log::info('PayPal Direct IPN verification response', [
                    'response' => $responseBody,
                    'ipn_data' => $ipnData
                ]);

                return $responseBody === 'VERIFIED';
            }

            Log::error('PayPal Direct IPN verification request failed', [
                'status' => $response->status(),
                'response' => $response->body(),
                'ipn_data' => $ipnData
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error('PayPal Direct IPN verification error: ' . $e->getMessage(), [
                'ipn_data' => $ipnData,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Process verified IPN payment data.
     *
     * @param array $ipnData
     * @return bool
     */
    protected function processIpnPayment(array $ipnData): bool
    {
        try {
            // Extract key data from IPN
            $transactionId = $ipnData['txn_id'] ?? null;
            $paymentStatus = $ipnData['payment_status'] ?? null;
            $receiverEmail = $ipnData['receiver_email'] ?? null;
            $amount = $ipnData['mc_gross'] ?? null;
            $currency = $ipnData['mc_currency'] ?? null;
            $custom = $ipnData['custom'] ?? null;

            // Validate required fields
            if (!$transactionId || !$paymentStatus || !$receiverEmail || !$amount || !$currency || !$custom) {
                Log::error('PayPal Direct IPN missing required fields', ['ipn_data' => $ipnData]);
                return false;
            }

            // Validate receiver email
            if (strtolower($receiverEmail) !== strtolower($this->getBusinessEmail())) {
                Log::error('PayPal Direct IPN receiver email mismatch', [
                    'expected' => $this->getBusinessEmail(),
                    'received' => $receiverEmail,
                    'ipn_data' => $ipnData
                ]);
                return false;
            }

            // Decode and validate custom token
            $tokenData = $this->decodeSecureToken($custom);
            if (!$tokenData) {
                Log::error('PayPal Direct IPN invalid custom token', ['custom' => $custom]);
                return false;
            }

            // Find the order and payment
            $order = Order::find($tokenData['order_id']);
            if (!$order) {
                Log::error('PayPal Direct IPN order not found', ['order_id' => $tokenData['order_id']]);
                return false;
            }

            $payment = Payment::find($tokenData['payment_id']);
            if (!$payment) {
                Log::error('PayPal Direct IPN payment not found', ['payment_id' => $tokenData['payment_id']]);
                return false;
            }

            // Validate amount and currency
            if ((float)$amount !== (float)$order->net_amount || strtoupper($currency) !== strtoupper($order->currency)) {
                Log::error('PayPal Direct IPN amount/currency mismatch', [
                    'expected_amount' => $order->net_amount,
                    'expected_currency' => $order->currency,
                    'received_amount' => $amount,
                    'received_currency' => $currency,
                    'ipn_data' => $ipnData
                ]);
                return false;
            }

            // Check if payment was already processed
            if ($payment->status === Payment::STATUS_COMPLETED) {
                Log::info('PayPal Direct IPN payment already processed', [
                    'payment_id' => $payment->id,
                    'transaction_id' => $transactionId
                ]);
                return true;
            }

            // Process payment based on status
            switch (strtolower($paymentStatus)) {
                case 'completed':
                    return $this->handleCompletedPayment($payment, $order, $transactionId, $ipnData);

                case 'pending':
                    return $this->handlePendingPayment($payment, $order, $transactionId, $ipnData);

                case 'failed':
                case 'denied':
                case 'expired':
                case 'voided':
                    return $this->handleFailedPayment($payment, $order, $transactionId, $ipnData);

                default:
                    Log::info('PayPal Direct IPN unhandled payment status', [
                        'status' => $paymentStatus,
                        'transaction_id' => $transactionId
                    ]);
                    return true;
            }
        } catch (\Exception $e) {
            Log::error('Error processing PayPal Direct IPN payment: ' . $e->getMessage(), [
                'ipn_data' => $ipnData,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Handle completed payment.
     *
     * @param Payment $payment
     * @param Order $order
     * @param string $transactionId
     * @param array $ipnData
     * @return bool
     */
    protected function handleCompletedPayment(Payment $payment, Order $order, string $transactionId, array $ipnData): bool
    {
        try {
            // Update payment record
            $this->updatePaymentRecord(
                $payment,
                Payment::STATUS_COMPLETED,
                $transactionId,
                ['ipn_data' => $ipnData]
            );

            // Update order
            $order->payment_status = 'paid';
            $order->payment_transaction_id = $transactionId;
            $order->payment_method = 'PayPal Direct';
            $order->order_status = 'bidding';
            $order->payment_date = now();
            $order->save();

            // Notify qualified writers about the new order available for bidding
            try {
                $writerNotificationService = new WriterNotificationService();
                $writerNotificationService->notifyQualifiedWriters($order);
            } catch (\Exception $e) {
                Log::error('Failed to notify writers about new order', [
                    'error' => $e->getMessage(),
                    'order_id' => $order->id
                ]);
            }

            // Send notifications
            $this->sendPaymentNotifications($order, $payment, true);

            Log::info('PayPal Direct payment completed successfully', [
                'order_id' => $order->id,
                'payment_id' => $payment->id,
                'transaction_id' => $transactionId
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Error handling completed PayPal Direct payment: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'payment_id' => $payment->id,
                'transaction_id' => $transactionId
            ]);
            return false;
        }
    }

    /**
     * Handle pending payment.
     *
     * @param Payment $payment
     * @param Order $order
     * @param string $transactionId
     * @param array $ipnData
     * @return bool
     */
    protected function handlePendingPayment(Payment $payment, Order $order, string $transactionId, array $ipnData): bool
    {
        try {
            // Update payment record
            $this->updatePaymentRecord(
                $payment,
                Payment::STATUS_PENDING,
                $transactionId,
                ['ipn_data' => $ipnData]
            );

            Log::info('PayPal Direct payment marked as pending', [
                'order_id' => $order->id,
                'payment_id' => $payment->id,
                'transaction_id' => $transactionId
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Error handling pending PayPal Direct payment: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'payment_id' => $payment->id,
                'transaction_id' => $transactionId
            ]);
            return false;
        }
    }

    /**
     * Handle failed payment.
     *
     * @param Payment $payment
     * @param Order $order
     * @param string $transactionId
     * @param array $ipnData
     * @return bool
     */
    protected function handleFailedPayment(Payment $payment, Order $order, string $transactionId, array $ipnData): bool
    {
        try {
            // Update payment record
            $this->updatePaymentRecord(
                $payment,
                Payment::STATUS_FAILED,
                $transactionId,
                ['ipn_data' => $ipnData]
            );

            // Send failure notifications
            $this->sendPaymentNotifications($order, $payment, false);

            Log::info('PayPal Direct payment marked as failed', [
                'order_id' => $order->id,
                'payment_id' => $payment->id,
                'transaction_id' => $transactionId,
                'status' => $ipnData['payment_status']
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Error handling failed PayPal Direct payment: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'payment_id' => $payment->id,
                'transaction_id' => $transactionId
            ]);
            return false;
        }
    }

    /**
     * Send payment notifications.
     *
     * @param Order $order
     * @param Payment $payment
     * @param bool $success
     * @return void
     */
    protected function sendPaymentNotifications(Order $order, Payment $payment, bool $success): void
    {
        try {
            if ($success) {
                // Notify user of successful payment
                if ($order->user) {
                    $order->user->notify(new PaymentSuccessfulNotification($order, $payment));
                }

                // Notify admins
                AdminNotificationService::notifyAllAdmins(
                    new AdminPaymentNotification($order, $payment, $order->user->name ?? 'Unknown User')
                );
            } else {
                // Notify user of failed payment
                if ($order->user) {
                    $order->user->notify(new PaymentFailedNotification($order, $payment));
                }

                // Notify admins
                AdminNotificationService::notifyAllAdmins(
                    new AdminPaymentNotification($order, $payment, $order->user->name ?? 'Unknown User')
                );
            }
        } catch (\Exception $e) {
            Log::error('Failed to send payment notifications', [
                'order_id' => $order->id,
                'payment_id' => $payment->id,
                'success' => $success,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Generate a secure token for the custom field.
     *
     * @param Order $order
     * @param Payment $payment
     * @return string
     */
    protected function generateSecureToken(Order $order, Payment $payment): string
    {
        $data = [
            'order_id' => $order->id,
            'payment_id' => $payment->id,
            'amount' => $order->net_amount,
            'currency' => $order->currency,
            'nonce' => Str::random(16),
            'timestamp' => now()->timestamp
        ];

        $signature = hash_hmac('sha256', json_encode($data), config('app.key'));
        $data['signature'] = $signature;

        return base64_encode(json_encode($data));
    }

    /**
     * Decode and validate the secure token.
     *
     * @param string $token
     * @return array|null
     */
    protected function decodeSecureToken(string $token): ?array
    {
        try {
            $decoded = json_decode(base64_decode($token), true);
            if (!$decoded) {
                return null;
            }

            // Verify signature
            $signature = $decoded['signature'] ?? null;
            unset($decoded['signature']);

            $expectedSignature = hash_hmac('sha256', json_encode($decoded), config('app.key'));

            if (!hash_equals($signature, $expectedSignature)) {
                Log::warning('PayPal Direct IPN signature mismatch', ['token' => $token]);
                return null;
            }

            // Check if token is not expired (24 hours)
            $tokenAge = now()->timestamp - ($decoded['timestamp'] ?? 0);
            if ($tokenAge > 86400) {
                Log::warning('PayPal Direct IPN token expired', ['token_age' => $tokenAge]);
                return null;
            }

            return $decoded;
        } catch (\Exception $e) {
            Log::error('Error decoding PayPal Direct secure token: ' . $e->getMessage(), ['token' => $token]);
            return null;
        }
    }

    /**
     * Get the business email for the current mode.
     */
    protected function getBusinessEmail(): string
    {
        $mode = $this->paymentMethod->getConfig('mode', 'sandbox');
        $configKey = $mode === 'sandbox' ? 'sandbox_business_email' : 'business_email';

        $email = $this->paymentMethod->getConfig($configKey);
        if (!$email) {
            Log::warning('PayPal Direct business email missing in DB config, checking fallback', [
                'mode' => $mode,
                'payment_method_id' => $this->paymentMethod->id,
                'config_data' => $this->paymentMethod->config_data,
            ]);
            $fallback = config("paypal_direct." . ($mode === 'sandbox' ? 'sandbox.business_email' : 'live.business_email'));
            if ($fallback) {
                return $fallback;
            }
            throw new \RuntimeException("PayPal Direct business email not configured for {$mode} mode");
        }

        return $email;
    }

    /**
     * Get the form action URL for the current mode with fallbacks.
     */
    protected function getFormAction(): string
    {
        $mode = $this->paymentMethod->getConfig('mode', 'sandbox');
        $configKey = "form_action_{$mode}";

        $url = $this->paymentMethod->getConfig($configKey);
        if (!$url) {
            Log::warning('PayPal Direct form action URL missing in DB config, checking fallback', [
                'mode' => $mode,
                'payment_method_id' => $this->paymentMethod->id,
                'config_data' => $this->paymentMethod->config_data,
            ]);
            $fallback = config('paypal_direct.' . ($mode === 'sandbox' ? 'sandbox.form_action' : 'live.form_action'));
            if ($fallback) {
                return $fallback;
            }
            // Final hardcoded fallback
            return $mode === 'sandbox'
                ? 'https://www.sandbox.paypal.com/cgi-bin/webscr'
                : 'https://www.paypal.com/cgi-bin/webscr';
        }

        return $url;
    }

    /**
     * Get the IPN verification URL for the current mode with fallbacks.
     */
    protected function getIpnVerifyUrl(): string
    {
        $mode = $this->paymentMethod->getConfig('mode', 'sandbox');
        $configKey = "ipn_verify_{$mode}";

        $url = $this->paymentMethod->getConfig($configKey);
        if (!$url) {
            Log::warning('PayPal Direct IPN verify URL missing in DB config, checking fallback', [
                'mode' => $mode,
                'payment_method_id' => $this->paymentMethod->id,
                'config_data' => $this->paymentMethod->config_data,
            ]);
            $fallback = config('paypal_direct.' . ($mode === 'sandbox' ? 'sandbox.ipn_verify' : 'live.ipn_verify'));
            if ($fallback) {
                return $fallback;
            }
            // Final hardcoded fallback
            return $mode === 'sandbox'
                ? 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr'
                : 'https://ipnpb.paypal.com/cgi-bin/webscr';
        }

        return $url;
    }
} 