<?php

namespace App\Services\Payments;

use App\Models\Order;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Services\WriterNotificationService;
use Illuminate\Support\Facades\Log;
use App\Services\AdminNotificationService;
use App\Notifications\AdminPaymentNotification;
use App\Notifications\PaymentFailedNotification;
use App\Notifications\PaymentRefundedNotification;
use App\Notifications\PaymentSuccessfulNotification;
use Srmklive\PayPal\Services\PayPal as PayPalClient;

class PayPalPaymentService extends AbstractPaymentService
{
    /**
     * @var PayPalClient
     */
    protected $paypalClient;

    /**
     * Create a new PayPal payment service instance.
     *
     * @param PaymentMethod $paymentMethod
     */
    public function __construct(PaymentMethod $paymentMethod)
    {
        parent::__construct($paymentMethod);

        // Initialize PayPal client
        $this->initializePayPalClient();
    }

    /**
     * Initialize the PayPal client with configuration.
     *
     * @return void
     */
    protected function initializePayPalClient(): void
    {
        $this->paypalClient = new PayPalClient;

        // Get configuration from payment method or fall back to config file
        $mode = $this->paymentMethod->getConfig('mode', config('paypal.mode', 'sandbox'));

        $paypalConfig = [
            'mode' => $mode,
            'sandbox' => [
                'client_id' => $this->paymentMethod->getConfig('client_id', config('paypal.sandbox_client_id')),
                'client_secret' => $this->paymentMethod->getConfig('client_secret', config('paypal.sandbox_client_secret')),
                'app_id' => $this->paymentMethod->getConfig('app_id', ''),
            ],
            'live' => [
                'client_id' => $this->paymentMethod->getConfig('live_client_id', config('paypal.live.client_id')),
                'client_secret' => $this->paymentMethod->getConfig('live_client_secret', config('paypal.live.client_secret')),
                'app_id' => $this->paymentMethod->getConfig('live_app_id', config('paypal.live.app_id')),
            ],
            'payment_action' => 'Sale',
            'currency' => 'USD',
            'notify_url' => config('paypal.notify_url'),
            'locale' => config('paypal.locale', 'en_US'),
            'validate_ssl' => config('paypal.validate_ssl', true),
        ];

        // For development environments, disable SSL verification if configured
        if (app()->environment('local') || env('PAYPAL_SSL_VERIFY') === false) {
            $paypalConfig['http.CURLOPT_SSL_VERIFYPEER'] = false;
            $paypalConfig['http.CURLOPT_SSL_VERIFYHOST'] = 0;
        }

        $this->paypalClient->setApiCredentials($paypalConfig);

        try {
            $this->paypalClient->getAccessToken();
        } catch (\Exception $e) {
            Log::error('PayPal authentication error: ' . $e->getMessage());
            throw new \RuntimeException('Failed to authenticate with PayPal: ' . $e->getMessage());
        }
    }

    /**
     * Initialize a payment for an order.
     *
     * @param Order $order
     * @return array Payment initialization data
     */
    public function initializePayment(Order $order): array
    {
        // Create a pending payment record
        $payment = $this->createPaymentRecord($order);

        try {
            $returnUrl = route('orders.show', $order->id);

            $data = [
                'intent' => 'CAPTURE',
                'application_context' => [
                    'return_url' => $returnUrl,
                    'cancel_url' => $returnUrl,
                ],
                'purchase_units' => [
                    [
                        'reference_id' => $order->order_number,
                        'description' => "Order #{$order->order_number}",
                        'amount' => [
                            'currency_code' => strtoupper($order->currency), // Use order's currency
                            'value' => number_format($order->net_amount, 2, '.', '')
                        ]
                    ]
                ]
            ];

            $response = $this->paypalClient->createOrder($data);
            Log::info('PayPal createOrder response: ', ['response' => $response]);

            if (!isset($response['id'])) {
                Log::error('PayPal missing order ID in response', ['response' => $response]);
                throw new \Exception('Invalid response from PayPal');
            }

            // Update payment record with PayPal order ID
            $this->updatePaymentRecord(
                $payment,
                Payment::STATUS_PENDING,
                $response['id'],
                ['paypal_order_data' => $response]
            );

            return [
                'success' => true,
                'payment_id' => $payment->id,
                'paypal_order_id' => $response['id'],
                'client_config' => $this->getClientConfig($order->currency) // Pass currency to client config
            ];
        } catch (\Exception $e) {
            $this->handlePaymentError($order, $payment, $e);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }






    /**
     * Process a payment after user confirmation.
     *
     * @param Order $order
     * @param array $paymentData
     * @return Payment
     */
    public function processPayment(Order $order, array $paymentData): Payment
    {
        // Find existing payment or create a new one
        $payment = Payment::where('order_id', $order->id)
            ->where('payment_method', $this->paymentMethod->name)
            ->where('status', Payment::STATUS_PENDING)
            ->latest()
            ->first();

        if (!$payment) {
            $payment = $this->createPaymentRecord($order);
        }

        try {
            // Verify the payment with PayPal
            $paypalOrderId = $paymentData['paymentId'] ?? null;

            if (!$paypalOrderId) {
                throw new \Exception('PayPal order ID is missing');
            }

            $paymentResult = $this->paypalClient->showOrderDetails($paypalOrderId);

            // Verify the payment status
            if ($paymentResult['status'] === 'COMPLETED') {
                // Get the payment details
                $amount = $paymentResult['purchase_units'][0]['amount']['value'] ?? 0;
                $currency = $paymentResult['purchase_units'][0]['amount']['currency_code'] ?? '';

                // Verify both amount and currency
                if (
                    (float)$amount === (float)$order->net_amount &&
                    strtoupper($currency) === strtoupper($order->currency)
                ) {

                    // Update the payment record
                    $payment = $this->updatePaymentRecord(
                        $payment,
                        Payment::STATUS_COMPLETED,
                        $paypalOrderId,
                        ['paypal_details' => $paymentResult]
                    );

                    // Update the order
                    $order->payment_status = 'paid';
                    $order->payment_transaction_id = $paypalOrderId;
                    $order->payment_method = 'PayPal';
                    $order->order_status = 'bidding';
                    $order->payment_date = now();
                    $order->save();

                    // Notify qualified writers about the new order available for bidding
                    try {
                        $writerNotificationService = new WriterNotificationService();
                        $writerNotificationService->notifyQualifiedWriters($order);
                    } catch (\Exception $e) {
                        Log::error('Failed to notify writers about new order', [
                            'error' => $e->getMessage(),
                            'order_id' => $order->id
                        ]);
                    }

                    return $payment;
                } else {
                    // Log the mismatch details for debugging
                    Log::error('PayPal payment verification failed', [
                        'order_id' => $order->id,
                        'expected_amount' => (float)$order->net_amount,
                        'actual_amount' => (float)$amount,
                        'expected_currency' => strtoupper($order->currency),
                        'actual_currency' => $currency
                    ]);

                    throw new \Exception('Payment amount or currency mismatch');
                }
            } else {
                throw new \Exception('Payment not completed: ' . ($paymentResult['status'] ?? 'Unknown status'));
            }
        } catch (\Exception $e) {
            return $this->handlePaymentError($order, $payment, $e);
        }
    }

    /**
     * Verify a payment using provider-specific methods.
     *
     * @param string $transactionId
     * @return bool
     */
    public function verifyPayment(string $transactionId): bool
    {
        try {
            $paymentDetails = $this->paypalClient->showOrderDetails($transactionId);
            return isset($paymentDetails['status']) && $paymentDetails['status'] === 'COMPLETED';
        } catch (\Exception $e) {
            Log::error('PayPal verification error: ' . $e->getMessage(), [
                'transaction_id' => $transactionId
            ]);
            return false;
        }
    }

    /**
     * Handle webhook events from PayPal.
     *
     * @param array $payload
     * @return bool
     */
    public function handleWebhook(array $payload): bool
    {
        Log::info('PayPal webhook received', ['event_type' => $payload['event_type'] ?? 'unknown']);

        try {
            // Verify webhook signature (implementation depends on PayPal's webhook signature verification)
            // This is a simplified version - in production, you should verify the webhook signature

            $eventType = $payload['event_type'] ?? '';
            $resource = $payload['resource'] ?? [];

            if (empty($eventType) || empty($resource)) {
                Log::error('Invalid PayPal webhook payload', ['payload' => $payload]);
                return false;
            }

            // Handle different event types
            switch ($eventType) {
                case 'PAYMENT.CAPTURE.COMPLETED':
                    return $this->handlePaymentCompleted($resource);

                case 'PAYMENT.CAPTURE.DENIED':
                case 'PAYMENT.CAPTURE.DECLINED':
                    return $this->handlePaymentFailed($resource);

                case 'PAYMENT.CAPTURE.REFUNDED':
                    return $this->handlePaymentRefunded($resource);

                default:
                    Log::info('Unhandled PayPal webhook event', ['event_type' => $eventType]);
                    return true; // Return true for unhandled events to acknowledge receipt
            }
        } catch (\Exception $e) {
            Log::error('Error processing PayPal webhook: ' . $e->getMessage(), [
                'payload' => $payload,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Handle a completed payment webhook event.
     *
     * @param array $resource
     * @return bool
     */
    // protected function handlePaymentCompleted(array $resource): bool
    // {
    //     $transactionId = $resource['id'] ?? null;

    //     if (!$transactionId) {
    //         Log::error('Missing transaction ID in PayPal webhook', ['resource' => $resource]);
    //         return false;
    //     }

    //     // Find the payment by transaction ID
    //     $payment = Payment::where('transaction_id', $transactionId)->first();

    //     if (!$payment) {
    //         // Try to find by custom ID or other reference in the resource
    //         $referenceId = $resource['purchase_units'][0]['reference_id'] ?? null;

    //         if ($referenceId) {
    //             $order = Order::where('order_number', $referenceId)->first();

    //             if ($order) {
    //                 $payment = Payment::where('order_id', $order->id)
    //                     ->where('payment_method', $this->paymentMethod->name)
    //                     ->latest()
    //                     ->first();
    //             }
    //         }

    //         if (!$payment) {
    //             Log::error('Payment not found for PayPal transaction', [
    //                 'transaction_id' => $transactionId,
    //                 'reference_id' => $referenceId ?? 'not provided'
    //             ]);
    //             return false;
    //         }
    //     }

    //     // Update the payment status
    //     $this->updatePaymentRecord(
    //         $payment,
    //         Payment::STATUS_COMPLETED,
    //         $transactionId,
    //         ['webhook_data' => $resource]
    //     );

    //     $order = $payment->order;
    //     if ($order && $order->user) {
    //         // Notify the user
    //         $order->user->notify(new PaymentSuccessfulNotification($order, $payment));

    //         // Notify admins
    //         AdminNotificationService::notifyAllAdmins(
    //             new AdminPaymentNotification($order, $payment, $order->user->name)
    //         );
    //     }


    //     return true;
    // }

    protected function handlePaymentCompleted(array $resource): bool
    {
        $transactionId = $resource['id'] ?? null;

        if (!$transactionId) {
            Log::error('Missing transaction ID in PayPal webhook', ['resource' => $resource]);
            return false;
        }

        // Find the payment by transaction ID
        $payment = Payment::where('transaction_id', $transactionId)->first();

        if (!$payment) {
            // Try to find by reference ID...
            $referenceId = $resource['purchase_units'][0]['reference_id'] ?? null;
            if ($referenceId) {
                $order = Order::where('order_number', $referenceId)->first();
                if ($order) {
                    $payment = Payment::where('order_id', $order->id)
                        ->where('payment_method', $this->paymentMethod->name)
                        ->latest()
                        ->first();
                }
            }

            if (!$payment) {
                Log::error('Payment not found for PayPal transaction', [
                    'transaction_id' => $transactionId,
                    'reference_id' => $referenceId ?? 'not provided'
                ]);
                return false;
            }
        }

        // Update the payment status
        $this->updatePaymentRecord(
            $payment,
            Payment::STATUS_COMPLETED,
            $transactionId,
            ['webhook_data' => $resource]
        );

        // Get the order and user
        $order = $payment->order;
        if ($order && $order->user) {
            try {
                // Notify the user
                $order->user->notify(new PaymentSuccessfulNotification($order, $payment));

                // Add debug logging to see if this part is executed
                Log::info('Sending admin notification for successful PayPal payment', [
                    'order_id' => $order->id,
                    'payment_id' => $payment->id
                ]);

                // Notify admins - explicitly catch errors
                try {
                    AdminNotificationService::notifyAllAdmins(
                        new AdminPaymentNotification($order, $payment, $order->user->name)
                    );

                    Log::info('Admin notification sent successfully for PayPal payment');
                } catch (\Exception $e) {
                    Log::error('Failed to send admin notification for PayPal payment', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            } catch (\Exception $e) {
                Log::error('Error sending payment notifications', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return true;
    }

    /**
     * Handle a failed payment webhook event.
     *
     * @param array $resource
     * @return bool
     */
    // protected function handlePaymentFailed(array $resource): bool
    // {
    //     $transactionId = $resource['id'] ?? null;

    //     if (!$transactionId) {
    //         Log::error('Missing transaction ID in PayPal webhook', ['resource' => $resource]);
    //         return false;
    //     }

    //     // Find the payment by transaction ID
    //     $payment = Payment::where('transaction_id', $transactionId)->first();

    //     if ($payment) {
    //         // Update the payment status
    //         $this->updatePaymentRecord(
    //             $payment,
    //             Payment::STATUS_FAILED,
    //             $transactionId,
    //             ['webhook_data' => $resource]
    //         );

    //         return true;
    //     }

    //     Log::error('Payment not found for PayPal transaction', ['transaction_id' => $transactionId]);
    //     return false;
    // }

    protected function handlePaymentFailed(array $resource): bool
    {
        $transactionId = $resource['id'] ?? null;

        if (!$transactionId) {
            Log::error('Missing transaction ID in PayPal webhook', ['resource' => $resource]);
            return false;
        }

        // Find the payment by transaction ID
        $payment = Payment::where('transaction_id', $transactionId)->first();

        if (!$payment) {
            // Try to find by reference ID...
            $referenceId = $resource['purchase_units'][0]['reference_id'] ?? null;
            if ($referenceId) {
                $order = Order::where('order_number', $referenceId)->first();
                if ($order) {
                    $payment = Payment::where('order_id', $order->id)
                        ->where('payment_method', $this->paymentMethod->name)
                        ->latest()
                        ->first();
                }
            }

            if (!$payment) {
                Log::error('Payment not found for PayPal transaction', ['transaction_id' => $transactionId]);
                return false;
            }
        }

        // Extract error message from the webhook payload
        $errorMessage = $resource['status_details']['reason'] ?? 'Payment failed';

        // Update the payment status
        $this->updatePaymentRecord(
            $payment,
            Payment::STATUS_FAILED,
            $transactionId,
            ['webhook_data' => $resource, 'error' => $errorMessage]
        );

        // Get the order and user
        $order = $payment->order;
        if ($order && $order->user) {
            try {
                // Notify the user
                $order->user->notify(new PaymentFailedNotification($order, $payment, $errorMessage));

                // Notify admins
                try {
                    AdminNotificationService::notifyAllAdmins(
                        new AdminPaymentNotification($order, $payment, $order->user->name, true, $errorMessage)
                    );

                    Log::info('Admin notification sent for failed PayPal payment', [
                        'order_id' => $order->id,
                        'payment_id' => $payment->id
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to send admin notification for failed PayPal payment', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            } catch (\Exception $e) {
                Log::error('Error sending payment failure notifications', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return true;
    }


    /**
     * Handle a refunded payment webhook event.
     *
     * @param array $resource
     * @return bool
     */
    // protected function handlePaymentRefunded(array $resource): bool
    // {
    //     $transactionId = $resource['id'] ?? null;

    //     if (!$transactionId) {
    //         Log::error('Missing transaction ID in PayPal webhook', ['resource' => $resource]);
    //         return false;
    //     }

    //     // Find the payment by transaction ID
    //     $payment = Payment::where('transaction_id', $transactionId)->first();

    //     if ($payment) {
    //         // Update the payment status
    //         $this->updatePaymentRecord(
    //             $payment,
    //             Payment::STATUS_REFUNDED,
    //             $transactionId,
    //             ['webhook_data' => $resource]
    //         );

    //         return true;
    //     }

    //     Log::error('Payment not found for PayPal transaction', ['transaction_id' => $transactionId]);
    //     return false;
    // }

    /**
     * Handle a refunded payment webhook event.
     *
     * @param array $resource
     * @return bool
     */
    protected function handlePaymentRefunded(array $resource): bool
    {
        $transactionId = $resource['id'] ?? null;

        if (!$transactionId) {
            Log::error('Missing transaction ID in PayPal webhook', ['resource' => $resource]);
            return false;
        }

        // Find the payment by transaction ID
        $payment = Payment::where('transaction_id', $transactionId)->first();

        if (!$payment) {
            // Try to find by reference ID...
            $referenceId = $resource['purchase_units'][0]['reference_id'] ?? null;
            if ($referenceId) {
                $order = Order::where('order_number', $referenceId)->first();
                if ($order) {
                    $payment = Payment::where('order_id', $order->id)
                        ->where('payment_method', $this->paymentMethod->name)
                        ->latest()
                        ->first();
                }
            }

            if (!$payment) {
                Log::error('Payment not found for PayPal refund transaction', ['transaction_id' => $transactionId]);
                return false;
            }
        }

        // Get refund amount
        $refundAmount = 0;
        $isFullRefund = false;

        // Try to extract the refund amount from the resource
        if (isset($resource['amount']['value'])) {
            $refundAmount = (float)$resource['amount']['value'];
            $isFullRefund = $refundAmount >= $payment->amount;
        } elseif (isset($resource['amount']['total'])) {
            $refundAmount = (float)$resource['amount']['total'];
            $isFullRefund = $refundAmount >= $payment->amount;
        } else {
            // If we can't determine the exact amount, use the full payment amount
            $refundAmount = $payment->amount;
            $isFullRefund = true;
        }

        // Get refund reason if available
        $reason = $resource['note_to_payer'] ?? $resource['status_details']['reason'] ?? '';

        // Update the payment status
        $newStatus = $isFullRefund ? Payment::STATUS_REFUNDED : Payment::STATUS_PARTIALLY_REFUNDED;
        $this->updatePaymentRecord(
            $payment,
            $newStatus,
            $payment->transaction_id, // Keep the original transaction ID
            ['webhook_data' => $resource, 'refund_amount' => $refundAmount]
        );

        // Update order payment status if order exists
        if ($payment->order) {
            $payment->order->updatePaymentStatusAfterRefund($payment, $refundAmount);

            Log::info('Updated order payment status after refund webhook', [
                'order_id' => $payment->order->id,
                'new_status' => $payment->order->payment_status
            ]);
        }

        // Get the order and user to send notifications
        $order = $payment->order;
        if ($order && $order->user) {
            try {
                // Notify the user about the refund
                $order->user->notify(new PaymentRefundedNotification(
                    $order,
                    $payment,
                    $refundAmount,
                    $isFullRefund,
                    $reason
                ));

                // Notify admins
                try {
                    AdminNotificationService::notifyAllAdmins(
                        new AdminPaymentNotification(
                            $order,
                            $payment,
                            $order->user->name,
                            false,
                            "Refund processed: " . ($isFullRefund ? "Full" : "Partial") . " - $refundAmount"
                        )
                    );

                    Log::info('Refund notifications sent for PayPal payment', [
                        'payment_id' => $payment->id,
                        'order_id' => $order->id,
                        'amount' => $refundAmount,
                        'is_full_refund' => $isFullRefund
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to send admin notification for PayPal refund', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            } catch (\Exception $e) {
                Log::error('Failed to send user notification for PayPal refund', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return true;
    }




    /**
     * Get the client-side configuration needed for PayPal.
     *
     * @param string|null $currency Currency code to use (defaults to USD)
     * @return array
     */
    public function getClientConfig(?string $currency = null): array
    {
        $mode = $this->paymentMethod->getConfig('mode', config('paypal.mode', 'sandbox'));
        $clientIdKey = $mode === 'sandbox' ? 'client_id' : 'live_client_id';

        // Use correct config key format
        $configKey = 'paypal.' . $mode . '_client_id';

        return [
            'clientId' => $this->paymentMethod->getConfig($clientIdKey, config($configKey)),
            'currency' => strtoupper($currency ?? 'USD'), // Use provided currency or default to USD
            'mode' => $mode
        ];
    }






    /**
     * Test connection with PayPal API.
     *
     * @return array
     */
    public function testConnection(): array
    {
        try {
            // Check if required config values exist
            $clientId = $this->paymentMethod->getConfig('client_id', config('paypal.sandbox_client_id'));
            $clientSecret = $this->paymentMethod->getConfig('client_secret', config('paypal.sandbox_client_secret'));

            if (empty($clientId) || empty($clientSecret)) {
                return [
                    'success' => false,
                    'message' => 'Missing required PayPal API credentials',
                ];
            }

            // Get API mode
            $mode = $this->paymentMethod->getConfig('mode', config('paypal.mode', 'sandbox'));

            // Use the existing PayPal client to test connection
            try {
                $this->paypalClient->getAccessToken();

                return [
                    'success' => true,
                    'message' => 'Successfully connected to PayPal API',
                    'details' => [
                        'environment' => $mode === 'sandbox' ? 'Sandbox' : 'Production',
                        'client_id' => substr($clientId, 0, 5) . '****' // Show only the beginning for security
                    ],
                ];
            } catch (\Exception $e) {
                Log::error('PayPal authentication error: ' . $e->getMessage());

                return [
                    'success' => false,
                    'message' => 'Failed to authenticate with PayPal API: ' . $e->getMessage(),
                ];
            }
        } catch (\Exception $e) {
            Log::error('PayPal connection test failed', [
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'PayPal connection test failed: ' . $e->getMessage(),
            ];
        }
    }





    /**
     * Process a refund for a payment.
     *
     * @param Payment $payment
     * @param float $amount
     * @param string $reason
     * @return array
     */
    public function processRefund(Payment $payment, float $amount, string $reason): array
    {
        try {
            // Verify the payment belongs to PayPal
            if ($payment->payment_method !== PaymentMethod::PAYPAL) {
                return [
                    'success' => false,
                    'message' => 'Invalid payment method for refund'
                ];
            }

            // Check if payment can be refunded using our enhanced method
            if (!$payment->canBeRefunded()) {
                return [
                    'success' => false,
                    'message' => 'This payment cannot be refunded. Current status: ' . $payment->status
                ];
            }

            // Validate refund amount against available refundable amount
            $refundableAmount = $payment->getRefundableAmount();
            if ($amount > $refundableAmount) {
                return [
                    'success' => false,
                    'message' => "Cannot refund more than the available amount ({$refundableAmount})"
                ];
            }

            // Original transaction ID (Order ID) is required for refund
            $orderId = $payment->transaction_id;
            if (empty($orderId)) {
                return [
                    'success' => false,
                    'message' => 'Missing transaction ID for refund'
                ];
            }

            // Format amount as a string with exactly 2 decimal places
            $formattedAmount = number_format($amount, 2, '.', '');

            // Generate invoice ID
            $invoiceId = 'refund-' . $payment->id . '-' . time();

            // Create a short note from the reason
            $note = substr($reason, 0, 255);

            // First, get the order details to find the capture ID
            try {
                $orderDetails = $this->paypalClient->showOrderDetails($orderId);

                Log::info('PayPal order details retrieved', [
                    'payment_id' => $payment->id,
                    'order_id' => $orderId
                ]);

                // Extract the capture ID from the order details
                $captureId = null;

                // Check payment metadata first - it might already have the capture ID
                if (!empty($payment->metadata) && isset($payment->metadata['paypal_details']['purchase_units'][0]['payments']['captures'][0]['id'])) {
                    $captureId = $payment->metadata['paypal_details']['purchase_units'][0]['payments']['captures'][0]['id'];
                }
                // If not in metadata, check the retrieved order details
                elseif (isset($orderDetails['purchase_units'][0]['payments']['captures'][0]['id'])) {
                    $captureId = $orderDetails['purchase_units'][0]['payments']['captures'][0]['id'];
                }

                if (empty($captureId)) {
                    Log::error('Could not find capture ID in PayPal order details', [
                        'payment_id' => $payment->id,
                        'order_id' => $orderId,
                        'order_details' => $orderDetails
                    ]);

                    return [
                        'success' => false,
                        'message' => 'Could not find the required capture ID for refund'
                    ];
                }

                Log::info('Found PayPal capture ID for refund', [
                    'payment_id' => $payment->id,
                    'order_id' => $orderId,
                    'capture_id' => $captureId
                ]);

                // Get the original payment currency
                $originalCurrency = null;

                // First check if we have the currency in the payment record
                if (!empty($payment->currency)) {
                    $originalCurrency = strtoupper($payment->currency);
                }
                // Then check if we can get it from the order
                elseif ($payment->order && !empty($payment->order->currency)) {
                    $originalCurrency = strtoupper($payment->order->currency);
                }
                // Finally check if we can extract it from the PayPal order details
                elseif (isset($orderDetails['purchase_units'][0]['amount']['currency_code'])) {
                    $originalCurrency = $orderDetails['purchase_units'][0]['amount']['currency_code'];
                }
                // Default to USD as last resort
                else {
                    $originalCurrency = 'USD';
                }

                // Log the currency we're using for the refund
                Log::info('Processing PayPal refund with currency', [
                    'payment_id' => $payment->id,
                    'amount' => $formattedAmount,
                    'currency' => $originalCurrency,
                    'capture_id' => $captureId
                ]);

                // Save the current currency setting
                $currentCurrency = $this->paypalClient->getCurrency();

                // Set the currency to match the original payment
                $this->paypalClient->setCurrency($originalCurrency);

                try {
                    // Now process the refund with the correct currency
                    $response = $this->paypalClient->refundCapturedPayment(
                        $captureId,
                        $invoiceId,
                        $formattedAmount,
                        $note
                    );

                    // Restore the original currency setting
                    $this->paypalClient->setCurrency($currentCurrency);

                    if (isset($response['id']) && isset($response['status']) && $response['status'] === 'COMPLETED') {
                        Log::info('PayPal refund processed successfully', [
                            'payment_id' => $payment->id,
                            'refund_id' => $response['id'],
                            'amount' => $formattedAmount,
                            'is_full_refund' => ($amount >= $payment->amount)
                        ]);

                        // Update payment status based on refund amount
                        $payment->updateStatusAfterRefund($amount);

                        // Update order payment status if order exists
                        if ($payment->order) {
                            $payment->order->updatePaymentStatusAfterRefund($payment, $amount);

                            Log::info('Updated order payment status after refund', [
                                'order_id' => $payment->order->id,
                                'new_status' => $payment->order->payment_status
                            ]);
                        }

                        return [
                            'success' => true,
                            'message' => 'Refund processed successfully',
                            'transaction_id' => $response['id'],
                            'details' => $response,
                            'is_full_refund' => ($amount >= $payment->amount)
                        ];
                    } else {
                        // Restore the original currency setting in case of error
                        $this->paypalClient->setCurrency($currentCurrency);

                        Log::error('PayPal refund failed', [
                            'payment_id' => $payment->id,
                            'response' => $response
                        ]);

                        return [
                            'success' => false,
                            'message' => 'PayPal refund failed: ' . ($response['message'] ?? 'Unknown error'),
                            'details' => $response
                        ];
                    }
                } catch (\Exception $e) {
                    // Restore the original currency setting in case of exception
                    $this->paypalClient->setCurrency($currentCurrency);
                    throw $e;
                }
            } catch (\Exception $e) {
                Log::error('Error retrieving PayPal order details: ' . $e->getMessage(), [
                    'payment_id' => $payment->id,
                    'order_id' => $orderId,
                    'trace' => $e->getTraceAsString()
                ]);

                return [
                    'success' => false,
                    'message' => 'Error retrieving PayPal order details: ' . $e->getMessage()
                ];
            }
        } catch (\Exception $e) {
            // Get response if available
            $response = null;
            if ($e instanceof \GuzzleHttp\Exception\RequestException) {
                try {
                    $response = json_decode($e->getResponse()->getBody(), true);
                } catch (\Exception $ex) {
                    // In case json_decode fails
                    $response = 'Could not parse response body';
                }
            }

            Log::error('PayPal refund error: ' . $e->getMessage(), [
                'payment_id' => $payment->id,
                'amount' => $amount,
                'transaction_id' => $orderId ?? 'unknown',
                'paypal_response' => $response,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => 'Error processing PayPal refund: ' . $e->getMessage()
            ];
        }
    }
}
