<?php

namespace App\Services\Payments;

use App\Models\PaymentMethod;
use InvalidArgumentException;
use App\Services\Payments\PayPalPaymentService;
use App\Services\Payments\PayPalDirectPaymentService;
use App\Services\OrderStatusService;

class PaymentServiceFactory
{
    /**
     * Create a payment service instance for the specified payment method.
     *
     * @param string $methodName
     * @return PaymentServiceInterface
     * @throws InvalidArgumentException
     */
    public static function create(string $methodName): PaymentServiceInterface
    {
        $paymentMethod = PaymentMethod::where('name', $methodName)
            ->where('is_active', true)
            ->first();

        if (!$paymentMethod) {
            throw new InvalidArgumentException("Payment method '{$methodName}' is not active or does not exist.");
        }

        switch ($methodName) {
            case PaymentMethod::PAYPAL:
                return new PayPalPaymentService($paymentMethod);

            case PaymentMethod::STRIPE:
                $orderStatusService = app(OrderStatusService::class);
                return new StripePaymentService($paymentMethod, $orderStatusService);

            case PaymentMethod::PAYPAL_DIRECT:
                $orderStatusService = app(OrderStatusService::class);
                return new PayPalDirectPaymentService($paymentMethod, $orderStatusService);

            default:
                throw new InvalidArgumentException("Unsupported payment method: {$methodName}");
        }
    }

    /**
     * Get all active payment services.
     *
     * @return array
     */
    public static function getAllActive(): array
    {
        $activeMethods = PaymentMethod::getActive();
        $services = [];

        foreach ($activeMethods as $method) {
            try {
                $services[$method->name] = self::create($method->name);
            } catch (InvalidArgumentException $e) {
                // Skip unsupported methods
                continue;
            }
        }

        return $services;
    }

    /**
     * Get list of available payment service types.
     *
     * @return array
     */
    public static function getAvailableServices(): array
    {
        return [
            PaymentMethod::PAYPAL => 'PayPal',
            PaymentMethod::STRIPE => 'Stripe',
            PaymentMethod::PAYPAL_DIRECT => 'PayPal Direct',
            // Add other supported payment methods
        ];
    }
}
