<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderRevision;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class RevisionService
{
    /**
     * Create a new revision for an order.
     *
     * @param Order $order
     * @param array $data
     * @return OrderRevision
     */
    public function createRevision(Order $order, array $data): OrderRevision
    {
        try {
            $revision = OrderRevision::create([
                'order_id' => $order->id,
                'requested_by' => Auth::id(),
                'assigned_to' => $data['assigned_to'] ?? null,
                'revision_type' => $data['revision_type'] ?? OrderRevision::REVISION_TYPE_CONTENT,
                'revision_priority' => $data['revision_priority'] ?? OrderRevision::PRIORITY_MEDIUM,
                'revision_section' => $data['revision_section'] ?? null,
                'revision_reason' => $data['revision_reason'] ?? null,
                'specific_instructions' => $data['specific_instructions'] ?? null,
                'revision_status' => OrderRevision::STATUS_REQUESTED,
                'revision_deadline' => $data['revision_deadline'] ?? $this->calculateRevisionDeadline($order, $data['revision_priority'] ?? OrderRevision::PRIORITY_MEDIUM),
                'admin_notes' => $data['revision_admin_notes'] ?? null,
                'revision_metadata' => $data['revision_metadata'] ?? [],
                'requested_at' => now(),
                'revision_number' => $order->getNextRevisionNumber(),
            ]);

            // Update order status to revision_requested if not already
            if ($order->order_status !== Order::ORDER_STATUS_REVISION_REQUESTED) {
                $order->update([
                    'order_status' => Order::ORDER_STATUS_REVISION_REQUESTED,
                    'revision_count' => $order->revision_count + 1,
                    'revision_notes' => $data['revision_reason'] ?? 'Revision requested',
                ]);
            }

            Log::info('Revision created successfully', [
                'order_id' => $order->id,
                'revision_id' => $revision->id,
                'revision_type' => $revision->revision_type,
                'priority' => $revision->revision_priority,
            ]);

            return $revision;

        } catch (\Exception $e) {
            Log::error('Failed to create revision', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
                'data' => $data,
            ]);
            throw $e;
        }
    }

    /**
     * Update the status of a revision.
     *
     * @param OrderRevision $revision
     * @param string $status
     * @param array $metadata
     * @return bool
     */
    public function updateRevisionStatus(OrderRevision $revision, string $status, array $metadata = []): bool
    {
        try {
            if (!$revision->canTransitionTo($status)) {
                throw new \InvalidArgumentException("Cannot transition revision from {$revision->revision_status} to {$status}");
            }

            $updateData = ['revision_status' => $status];
            $timestampField = $this->getTimestampFieldForStatus($status);

            if ($timestampField) {
                $updateData[$timestampField] = now();
            }

            // Update revision
            $revision->update($updateData);

            // Update metadata if provided
            if (!empty($metadata)) {
                $currentMetadata = $revision->revision_metadata ?? [];
                $revision->update([
                    'revision_metadata' => array_merge($currentMetadata, $metadata)
                ]);
            }

            // If revision is completed, check if order should be updated
            if ($status === OrderRevision::STATUS_REVIEWED) {
                $this->handleRevisionCompletion($revision);
            }

            Log::info('Revision status updated', [
                'revision_id' => $revision->id,
                'order_id' => $revision->order_id,
                'old_status' => $revision->getOriginal('revision_status'),
                'new_status' => $status,
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to update revision status', [
                'revision_id' => $revision->id,
                'status' => $status,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Assign a revision to a writer.
     *
     * @param OrderRevision $revision
     * @param User $writer
     * @return bool
     */
    public function assignRevisionToWriter(OrderRevision $revision, User $writer): bool
    {
        try {
            // Check if writer is available and qualified
            if (!$this->isWriterQualifiedForRevision($writer, $revision)) {
                throw new \InvalidArgumentException("Writer is not qualified for this revision");
            }

            $revision->update([
                'assigned_to' => $writer->id,
                'revision_status' => OrderRevision::STATUS_ACKNOWLEDGED,
                'revision_acknowledged_at' => now(),
            ]);

            Log::info('Revision assigned to writer', [
                'revision_id' => $revision->id,
                'writer_id' => $writer->id,
                'order_id' => $revision->order_id,
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to assign revision to writer', [
                'revision_id' => $revision->id,
                'writer_id' => $writer->id,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Calculate revision deadline based on priority.
     *
     * @param Order $order
     * @param string $priority
     * @return Carbon
     */
    public function calculateRevisionDeadline(Order $order, string $priority): Carbon
    {
        $baseHours = match($priority) {
            OrderRevision::PRIORITY_HIGH => 4,
            OrderRevision::PRIORITY_MEDIUM => 8,
            OrderRevision::PRIORITY_LOW => 24,
            default => 8,
        };

        // Start with base deadline
        $deadline = now()->addHours($baseHours);
        
        // Check if we need to adjust based on order deadlines
        if ($order->writer_deadline && $deadline->isAfter($order->writer_deadline)) {
            $deadline = $order->writer_deadline->subHours(2); // 2 hours buffer
        }

        if ($order->deadline && $deadline->isAfter($order->deadline)) {
            $deadline = $order->deadline->subHours(4); // 4 hours buffer
        }

        // Ensure deadline is not in the past
        if ($deadline->isBefore(now())) {
            $deadline = now()->addHours(1);
        }

        // For testing purposes, let's ensure the deadline is always before the order deadlines
        if ($order->writer_deadline && $deadline->isAfter($order->writer_deadline)) {
            $deadline = $order->writer_deadline->subHours(1);
        }

        if ($order->deadline && $deadline->isAfter($order->deadline)) {
            $deadline = $order->deadline->subHours(1);
        }

        return $deadline;
    }

    /**
     * Get revisions that are overdue.
     *
     * @param Order $order
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getOverdueRevisions(Order $order)
    {
        return $order->overdueRevisions()->get();
    }

    /**
     * Get high priority revisions for an order.
     *
     * @param Order $order
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getHighPriorityRevisions(Order $order)
    {
        return $order->highPriorityRevisions()->get();
    }

    /**
     * Check if a writer is qualified for a specific revision.
     *
     * @param User $writer
     * @param OrderRevision $revision
     * @return bool
     */
    private function isWriterQualifiedForRevision(User $writer, OrderRevision $revision): bool
    {
        // Basic checks - can be expanded based on business rules
        if (!$writer->isWriter()) {
            return false;
        }

        // Check if writer is already assigned to this order
        if ($revision->order->writer_id === $writer->id) {
            return true;
        }

        // Check writer's current workload
        $activeRevisions = OrderRevision::where('assigned_to', $writer->id)
            ->whereIn('revision_status', [
                OrderRevision::STATUS_ACKNOWLEDGED,
                OrderRevision::STATUS_IN_PROGRESS
            ])
            ->count();

        return $activeRevisions < 3; // Limit to 3 active revisions per writer
    }

    /**
     * Get the timestamp field name for a given status.
     *
     * @param string $status
     * @return string|null
     */
    private function getTimestampFieldForStatus(string $status): ?string
    {
        return match($status) {
            OrderRevision::STATUS_ACKNOWLEDGED => 'revision_acknowledged_at',
            OrderRevision::STATUS_IN_PROGRESS => 'revision_started_at',
            OrderRevision::STATUS_SUBMITTED => 'revision_submitted_at',
            OrderRevision::STATUS_REVIEWED => 'revision_reviewed_at',
            default => null,
        };
    }

    /**
     * Handle revision completion and update order status if needed.
     *
     * @param OrderRevision $revision
     * @return void
     */
    private function handleRevisionCompletion(OrderRevision $revision): void
    {
        $order = $revision->order;

        // Refresh the order to ensure we have the latest data
        $order->refresh();

        // Check if all revisions are completed
        $activeRevisions = $order->revisions()
            ->whereIn('revision_status', [
                OrderRevision::STATUS_REQUESTED,
                OrderRevision::STATUS_ACKNOWLEDGED,
                OrderRevision::STATUS_IN_PROGRESS
            ])
            ->count();
        

        
        if ($activeRevisions === 0) {
            // All revisions completed, move order back to submitted status
            $order->update([
                'order_status' => Order::ORDER_STATUS_SUBMITTED,
                'work_submitted_at' => now(),
            ]);

            Log::info('Order moved back to submitted status after revision completion', [
                'order_id' => $order->id,
                'revision_id' => $revision->id,
            ]);
        }
    }

    /**
     * Get revision statistics for an order.
     *
     * @param Order $order
     * @return array
     */
    public function getRevisionStatistics(Order $order): array
    {
        $revisions = $order->revisions;
        
        return [
            'total_revisions' => $revisions->count(),
            'completed_revisions' => $revisions->where('revision_status', OrderRevision::STATUS_REVIEWED)->count(),
            'active_revisions' => $revisions->whereIn('revision_status', [
                OrderRevision::STATUS_REQUESTED,
                OrderRevision::STATUS_ACKNOWLEDGED,
                OrderRevision::STATUS_IN_PROGRESS
            ])->count(),
            'overdue_revisions' => $revisions->where('revision_deadline', '<', now())
                ->whereIn('revision_status', [
                    OrderRevision::STATUS_REQUESTED,
                    OrderRevision::STATUS_ACKNOWLEDGED,
                    OrderRevision::STATUS_IN_PROGRESS
                ])->count(),
            'high_priority_revisions' => $revisions->where('revision_priority', OrderRevision::PRIORITY_HIGH)->count(),
            'revision_types' => $revisions->groupBy('revision_type')->map->count(),
            'average_completion_time' => $this->calculateAverageCompletionTime($revisions),
        ];
    }

    /**
     * Calculate average completion time for revisions.
     *
     * @param \Illuminate\Database\Eloquent\Collection $revisions
     * @return float|null
     */
    private function calculateAverageCompletionTime($revisions): ?float
    {
        $completedRevisions = $revisions->where('revision_status', OrderRevision::STATUS_REVIEWED)
            ->whereNotNull('revision_reviewed_at')
            ->whereNotNull('requested_at');

        if ($completedRevisions->isEmpty()) {
            return null;
        }

        $totalHours = $completedRevisions->sum(function ($revision) {
            return $revision->requested_at->diffInHours($revision->revision_reviewed_at);
        });

        return round($totalHours / $completedRevisions->count(), 2);
    }
} 