<?php

namespace App\Services;

use App\Models\PageSeo;
use App\Models\BlogPost;
use Illuminate\Support\Facades\DB;

class SEOService
{
    /**
     * Create or update SEO data for a page
     */
    public function createOrUpdateSeo(int $pageId, array $seoData): PageSeo
    {
        return PageSeo::updateOrCreate(
            ['page_id' => $pageId],
            $seoData
        );
    }

    /**
     * Get SEO data for a page
     */
    public function getPageSeo(int $pageId): ?PageSeo
    {
        return PageSeo::where('page_id', $pageId)->first();
    }

    /**
     * Generate meta title from page title and site name
     */
    public function generateMetaTitle(string $pageTitle, string $siteName = 'Academic Scribe'): string
    {
        $maxLength = 60;
        $title = $pageTitle . ' | ' . $siteName;
        
        if (strlen($title) > $maxLength) {
            $pageTitleLength = $maxLength - strlen(' | ' . $siteName);
            $title = substr($pageTitle, 0, $pageTitleLength) . '... | ' . $siteName;
        }

        return $title;
    }

    /**
     * Generate meta description from content
     */
    public function generateMetaDescription(string $content, int $maxLength = 160): string
    {
        // Strip HTML tags and clean up
        $description = strip_tags($content);
        $description = preg_replace('/\s+/', ' ', $description);
        $description = trim($description);

        if (strlen($description) <= $maxLength) {
            return $description;
        }

        // Find the last complete word within the limit
        $truncated = substr($description, 0, $maxLength);
        $lastSpace = strrpos($truncated, ' ');
        
        if ($lastSpace !== false) {
            $description = substr($description, 0, $lastSpace);
        } else {
            $description = $truncated;
        }

        return $description . '...';
    }

    /**
     * Generate structured data for different page types
     */
    public function generateStructuredData(string $pageType, array $pageData, array $seoData = []): array
    {
        $structuredData = [];

        switch ($pageType) {
            case 'service':
                $structuredData = $this->generateServiceStructuredData($pageData, $seoData);
                break;
            case 'homepage':
                $structuredData = $this->generateHomepageStructuredData($pageData, $seoData);
                break;
            case 'blog':
                $structuredData = $this->generateBlogStructuredData($pageData, $seoData);
                break;
            case 'contact':
                $structuredData = $this->generateContactStructuredData($pageData, $seoData);
                break;
            default:
                $structuredData = $this->generateDefaultStructuredData($pageData, $seoData);
        }

        return $structuredData;
    }

    /**
     * Generate service page structured data
     */
    private function generateServiceStructuredData(array $pageData, array $seoData): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'Service',
            'name' => $pageData['title'] ?? '',
            'description' => $seoData['meta_description'] ?? '',
            'provider' => [
                '@type' => 'Organization',
                'name' => 'Academic Scribe',
                'url' => config('app.url'),
                'logo' => config('app.url') . '/images/logo.png'
            ],
            'serviceType' => $pageData['title'] ?? '',
            'areaServed' => 'Worldwide',
            'hasOfferCatalog' => [
                '@type' => 'OfferCatalog',
                'name' => 'Academic Writing Services',
                'itemListElement' => $this->getServiceOffers()
            ]
        ];
    }

    /**
     * Generate homepage structured data
     */
    private function generateHomepageStructuredData(array $pageData, array $seoData): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            'name' => 'Academic Scribe',
            'url' => config('app.url'),
            'logo' => config('app.url') . '/images/logo.png',
            'description' => $seoData['meta_description'] ?? '',
            'sameAs' => [
                'https://twitter.com/academicscribe',
                'https://linkedin.com/company/academic-scribe'
            ],
            'contactPoint' => [
                '@type' => 'ContactPoint',
                'telephone' => '+1-555-0123',
                'contactType' => 'customer service',
                'availableLanguage' => 'English'
            ],
            'address' => [
                '@type' => 'PostalAddress',
                'addressCountry' => 'US'
            ]
        ];
    }

    /**
     * Generate blog post structured data
     */
    private function generateBlogStructuredData(array $pageData, array $seoData): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'BlogPosting',
            'headline' => $pageData['title'] ?? '',
            'description' => $seoData['meta_description'] ?? '',
            'author' => [
                '@type' => 'Organization',
                'name' => 'Academic Scribe'
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => 'Academic Scribe',
                'logo' => [
                    '@type' => 'ImageObject',
                    'url' => config('app.url') . '/images/logo.png'
                ]
            ],
            'datePublished' => $pageData['published_at'] ?? now()->toISOString(),
            'dateModified' => $pageData['updated_at'] ?? now()->toISOString(),
            'mainEntityOfPage' => [
                '@type' => 'WebPage',
                '@id' => $pageData['canonical_url'] ?? ''
            ]
        ];
    }

    /**
     * Generate contact page structured data
     */
    private function generateContactStructuredData(array $pageData, array $seoData): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'ContactPage',
            'name' => $pageData['title'] ?? 'Contact Us',
            'description' => $seoData['meta_description'] ?? '',
            'mainEntity' => [
                '@type' => 'Organization',
                'name' => 'Academic Scribe',
                'contactPoint' => [
                    '@type' => 'ContactPoint',
                    'telephone' => '+1-555-0123',
                    'contactType' => 'customer service',
                    'email' => 'info@academicscribe.com'
                ]
            ]
        ];
    }

    /**
     * Generate default structured data
     */
    private function generateDefaultStructuredData(array $pageData, array $seoData): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'WebPage',
            'name' => $pageData['title'] ?? '',
            'description' => $seoData['meta_description'] ?? '',
            'url' => $pageData['canonical_url'] ?? '',
            'mainEntity' => [
                '@type' => 'Organization',
                'name' => 'Academic Scribe'
            ]
        ];
    }

    /**
     * Get service offers for structured data
     */
    private function getServiceOffers(): array
    {
        return [
            [
                '@type' => 'Offer',
                'itemOffered' => [
                    '@type' => 'Service',
                    'name' => 'Essay Writing'
                ]
            ],
            [
                '@type' => 'Offer',
                'itemOffered' => [
                    '@type' => 'Service',
                    'name' => 'Research Papers'
                ]
            ],
            [
                '@type' => 'Offer',
                'itemOffered' => [
                    '@type' => 'Service',
                    'name' => 'Dissertations'
                ]
            ]
        ];
    }

    /**
     * Generate Open Graph data from database SEO data
     * Returns data structure matching frontend expectations (keys without colons)
     */
    public function generateOpenGraphData(array $pageData, array $seoData): array
    {
        return [
            'title' => $seoData['og_title'] ?? $seoData['meta_title'] ?? $pageData['title'] ?? '',
            'description' => $seoData['og_description'] ?? $seoData['meta_description'] ?? '',
            'image' => $seoData['og_image'] ?? null,
            'image_alt' => $seoData['og_image_alt'] ?? null,
            'type' => $seoData['og_type'] ?? 'website',
            'url' => $seoData['og_url'] ?? $seoData['canonical_url'] ?? null,
            'site_name' => $seoData['og_site_name'] ?? 'Academic Scribe',
            'locale' => $seoData['og_locale'] ?? 'en_US',
        ];
    }

    /**
     * Generate Twitter Card data from database SEO data
     * Returns data structure matching frontend expectations (keys without colons)
     */
    public function generateTwitterCardData(array $pageData, array $seoData): array
    {
        return [
            'card' => $seoData['twitter_card'] ?? 'summary_large_image',
            'site' => $seoData['twitter_site'] ?? null,
            'creator' => $seoData['twitter_creator'] ?? null,
            'title' => $seoData['twitter_title'] ?? $seoData['og_title'] ?? $seoData['meta_title'] ?? null,
            'description' => $seoData['twitter_description'] ?? $seoData['og_description'] ?? $seoData['meta_description'] ?? null,
            'image' => $seoData['twitter_image'] ?? $seoData['og_image'] ?? null,
            'image_alt' => $seoData['twitter_image_alt'] ?? null,
        ];
    }

    /**
     * Update SEO data when blog posts are updated
     */
    public function updateRelatedSeoFromBlog(BlogPost $blogPost): void
    {
        // Find pages that reference this blog post
        $relatedPages = DB::table('page_content')
            ->whereRaw('JSON_SEARCH(content, "one", ?) IS NOT NULL', ['%' . $blogPost->id . '%'])
            ->orWhereRaw('JSON_SEARCH(content, "one", ?) IS NOT NULL', ['%' . $blogPost->slug . '%'])
            ->get();

        foreach ($relatedPages as $contentBlock) {
            // Update the page's SEO data with fresh blog post information
            $seo = PageSeo::where('page_id', $contentBlock->page_id)->first();
            
            if ($seo) {
                // Update structured data to include latest blog post info
                $structuredData = json_decode($seo->structured_data, true) ?? [];
                
                // Add or update blog post reference in structured data
                if (!isset($structuredData['mentions'])) {
                    $structuredData['mentions'] = [];
                }

                $structuredData['mentions'][] = [
                    '@type' => 'BlogPosting',
                    'headline' => $blogPost->title,
                    'url' => route('blog.show', $blogPost->slug),
                    'datePublished' => $blogPost->published_at,
                    'dateModified' => $blogPost->updated_at
                ];

                $seo->update([
                    'structured_data' => json_encode($structuredData)
                ]);
            }
        }
    }

    /**
     * Validate SEO data
     */
    public function validateSeoData(array $seoData): array
    {
        $errors = [];

        // Meta title validation
        if (isset($seoData['meta_title']) && strlen($seoData['meta_title']) > 60) {
            $errors[] = 'Meta title should be 60 characters or less';
        }

        // Meta description validation
        if (isset($seoData['meta_description']) && strlen($seoData['meta_description']) > 160) {
            $errors[] = 'Meta description should be 160 characters or less';
        }

        // URL validation
        if (isset($seoData['canonical_url']) && !filter_var($seoData['canonical_url'], FILTER_VALIDATE_URL)) {
            $errors[] = 'Canonical URL must be a valid URL';
        }

        // Structured data validation
        if (isset($seoData['structured_data']) && !is_array($seoData['structured_data'])) {
            $errors[] = 'Structured data must be a valid array';
        }

        return $errors;
    }

    /**
     * Get SEO suggestions for a page
     */
    public function getSeoSuggestions(array $pageData, array $seoData): array
    {
        $suggestions = [];

        // Check meta title length
        if (empty($seoData['meta_title'])) {
            $suggestions[] = 'Add a meta title for better SEO';
        } elseif (strlen($seoData['meta_title']) < 30) {
            $suggestions[] = 'Meta title could be longer (30-60 characters recommended)';
        } elseif (strlen($seoData['meta_title']) > 60) {
            $suggestions[] = 'Meta title is too long (60 characters maximum)';
        }

        // Check meta description
        if (empty($seoData['meta_description'])) {
            $suggestions[] = 'Add a meta description for better SEO';
        } elseif (strlen($seoData['meta_description']) < 120) {
            $suggestions[] = 'Meta description could be longer (120-160 characters recommended)';
        } elseif (strlen($seoData['meta_description']) > 160) {
            $suggestions[] = 'Meta description is too long (160 characters maximum)';
        }

        // Check for Open Graph image
        if (empty($seoData['og_image'])) {
            $suggestions[] = 'Add an Open Graph image for better social media sharing';
        }

        // Check for structured data
        if (empty($seoData['structured_data'])) {
            $suggestions[] = 'Add structured data for rich snippets';
        }

        return $suggestions;
    }
}
