<?php

namespace App\Services;

use App\Models\Page;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use InvalidArgumentException;
use RuntimeException;

class ServicePageScaffolder
{
    public function __construct(private readonly PageService $pageService)
    {
    }

    /**
     * Scaffold a new service page using the canonical template.
     *
     * @param  string  $slug
     * @param  array  $overrides
     * @return array{page:\App\Models\Page, blocks:array, seo:array}
     */
    public function scaffold(string $slug, array $overrides = []): array
    {
        $slug = Str::slug($slug);

        if (Page::where('slug', $slug)->exists()) {
            throw new InvalidArgumentException("A page with slug '{$slug}' already exists.");
        }

        $template = $this->loadTemplate();

        $title = $overrides['title'] ?? $overrides['page']['title'] ?? Str::headline(str_replace('-', ' ', $slug));

        $replacements = [
            '{{slug}}' => $slug,
            '{{title}}' => $title,
        ];

        $template = $this->replacePlaceholders($template, $replacements);

        $pageData = array_replace_recursive(
            $template['page'] ?? [],
            $overrides['page'] ?? [],
            [
                'title' => $title,
                'slug' => $slug,
            ]
        );

        $seoData = array_replace_recursive($template['seo'] ?? [], $overrides['seo'] ?? []);
        $blocks = $this->mergeBlocks($template['blocks'] ?? [], $overrides['blocks'] ?? []);

        $page = $this->pageService->createPage($pageData, $blocks, $seoData);

        return [
            'page' => $page,
            'blocks' => $blocks,
            'seo' => $seoData,
        ];
    }

    private function loadTemplate(): array
    {
        $path = resource_path('content/templates/service-page.json');

        if (!File::exists($path)) {
            throw new RuntimeException('Service page template not found at ' . $path);
        }

        $raw = File::get($path);
        $decoded = json_decode($raw, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new RuntimeException('Invalid JSON in service page template: ' . json_last_error_msg());
        }

        return $decoded;
    }

    private function replacePlaceholders(mixed $value, array $replacements): mixed
    {
        if (is_array($value)) {
            foreach ($value as $key => $item) {
                $value[$key] = $this->replacePlaceholders($item, $replacements);
            }

            return $value;
        }

        if (is_string($value)) {
            return str_replace(array_keys($replacements), array_values($replacements), $value);
        }

        return $value;
    }

    private function mergeBlocks(array $templateBlocks, array $overrideBlocks): array
    {
        $overridesByKey = collect($overrideBlocks)->keyBy('block_key');
        $merged = [];

        foreach ($templateBlocks as $index => $block) {
            $blockKey = $block['block_key'] ?? null;

            if (!$blockKey) {
                continue;
            }

            $override = $overridesByKey->get($blockKey, []);
            $block = array_replace_recursive($block, $override);
            $block['order'] = $block['order'] ?? ($index + 1);
            $block['is_active'] = $block['is_active'] ?? false;
            $block['content'] = $this->normaliseContent($block['content'] ?? []);

            $merged[] = $block;
        }

        return $merged;
    }

    private function normaliseContent(array $content): array
    {
        // Ensure nested arrays are well-formed
        if (isset($content['keyPoints']) && is_array($content['keyPoints'])) {
            $content['keyPoints'] = array_map(fn ($item) => $this->normaliseTextItem($item), $content['keyPoints']);
        }

        if (isset($content['trustSignals']) && is_array($content['trustSignals'])) {
            $content['trustSignals'] = array_map(fn ($item) => $this->normaliseTextItem($item), $content['trustSignals']);
        }

        if (isset($content['benefits']) && is_array($content['benefits'])) {
            $content['benefits'] = array_map(fn ($item) => $this->normaliseTextItem($item), $content['benefits']);
        }

        if (isset($content['guarantees']) && is_array($content['guarantees'])) {
            $content['guarantees'] = array_map(fn ($item) => $this->normaliseTextItem($item), $content['guarantees']);
        }

        if (isset($content['links']) && is_array($content['links'])) {
            $content['links'] = array_map(function ($link) {
                if (is_string($link)) {
                    return [
                        'title' => $link,
                        'url' => '#',
                        'description' => $link,
                        'description_html' => '<p>' . e($link) . '</p>',
                        'is_active' => true,
                    ];
                }

                $link['is_active'] = $link['is_active'] ?? true;
                $link['description_html'] = $link['description_html'] ?? ($link['description'] ?? null);

                return $link;
            }, $content['links']);
        }

        if (isset($content['sections']) && is_array($content['sections'])) {
            $content['sections'] = array_map(function ($section) {
                if (is_string($section)) {
                    return [
                        'title' => null,
                        'body_html' => $section,
                        'body' => $section,
                        'is_active' => true,
                    ];
                }

                $section['title'] = $section['title'] ?? null;
                $section['body_html'] = $section['body_html'] ?? ($section['body'] ?? '');
                $section['body'] = $section['body_html'];
                $section['is_active'] = $section['is_active'] ?? true;

                return $section;
            }, $content['sections']);
        }

        if (isset($content['preferredHeight'])) {
            $content['preferredHeight'] = (int) $content['preferredHeight'] ?: 480;
        }

        if (!isset($content['description_html']) && isset($content['description'])) {
            $content['description_html'] = $content['description'];
        }

        return $content;
    }

    private function normaliseTextItem(mixed $item): array
    {
        if (is_string($item)) {
            return [
                'text' => $item,
                'is_active' => true,
            ];
        }

        if (is_array($item)) {
            $item['text'] = $item['text'] ?? ($item['title'] ?? '');
            $item['is_active'] = $item['is_active'] ?? true;
        }

        return $item;
    }
}
