<?php

namespace App\Services;

use App\Models\Order;
use App\Models\User;
use App\Notifications\UrgentOrderNotification;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class UrgentOrderNotificationService
{
    const URGENT_48_HOURS = 48;
    const URGENT_24_HOURS = 24;
    const URGENT_12_HOURS = 12;

    /**
     * Get urgent orders with their urgency levels
     */
    public function getUrgentOrders(): array
    {
        $now = Carbon::now();
        
        $urgentOrders = [
            'overdue' => $this->getOverdueOrders($now),
            'critical_12h' => $this->getOrdersByDeadline($now, self::URGENT_12_HOURS, 'critical'),
            'urgent_24h' => $this->getOrdersByDeadline($now, self::URGENT_24_HOURS, 'urgent'),
            'warning_48h' => $this->getOrdersByDeadline($now, self::URGENT_48_HOURS, 'warning'),
        ];

        // Ensure all urgency levels are present and are arrays
        return [
            'overdue' => is_array($urgentOrders['overdue']) ? $urgentOrders['overdue'] : [],
            'critical_12h' => is_array($urgentOrders['critical_12h']) ? $urgentOrders['critical_12h'] : [],
            'urgent_24h' => is_array($urgentOrders['urgent_24h']) ? $urgentOrders['urgent_24h'] : [],
            'warning_48h' => is_array($urgentOrders['warning_48h']) ? $urgentOrders['warning_48h'] : []
        ];
    }

    /**
     * Get overdue orders
     */
    private function getOverdueOrders(Carbon $now): array
    {
        return Order::where('deadline', '<', $now)
            ->whereIn('order_status', Order::ACTIVE_STATUSES)
            ->with(['user', 'writer'])
            ->get()
            ->map(function ($order) {
                $order->urgency_level = 'overdue';
                $order->hours_remaining = $this->calculateHoursRemaining($order->deadline);
                $order->urgency_color = '#dc2626'; // Red for overdue
                return $order;
            })
            ->toArray();
    }

    /**
     * Get orders by deadline threshold
     */
    private function getOrdersByDeadline(Carbon $now, int $hours, string $urgencyLevel): array
    {
        $deadline = $now->copy()->addHours($hours);
        
        return Order::where('deadline', '<=', $deadline)
            ->whereIn('order_status', Order::ACTIVE_STATUSES)
            ->with(['user', 'writer'])
            ->get()
            ->map(function ($order) use ($urgencyLevel) {
                $order->urgency_level = $urgencyLevel;
                $order->hours_remaining = $this->calculateHoursRemaining($order->deadline);
                $order->urgency_color = $this->getUrgencyColor($urgencyLevel);
                return $order;
            })
            ->toArray();
    }

    /**
     * Calculate hours remaining until deadline
     */
    private function calculateHoursRemaining($deadline): float
    {
        $hours = Carbon::now()->diffInHours($deadline, false);
        return $hours;
    }

    /**
     * Get color for urgency level
     */
    private function getUrgencyColor(string $urgencyLevel): string
    {
        return match ($urgencyLevel) {
            'overdue' => '#dc2626',  // Red
            'critical' => '#dc2626', // Red
            'urgent' => '#ea580c',   // Orange
            'warning' => '#ca8a04',  // Yellow
            default => '#6b7280',    // Gray
        };
    }

    /**
     * Send notifications to admins and assigned writers
     */
    public function sendUrgentOrderNotifications(): void
    {
        $urgentOrders = $this->getUrgentOrders();
        
        // Send notifications to admins
        $this->notifyAdmins($urgentOrders);
        
        // Send notifications to assigned writers
        $this->notifyAssignedWriters($urgentOrders);
    }

    /**
     * Notify all admins about urgent orders
     */
    private function notifyAdmins(array $urgentOrders): void
    {
        $admins = User::where('user_type', 'admin')->get();
        
        foreach ($admins as $admin) {
            try {
                $admin->notify(new UrgentOrderNotification($urgentOrders, 'admin'));
            } catch (\Exception $e) {
                Log::error('Failed to send urgent order notification to admin', [
                    'admin_id' => $admin->id,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * Notify assigned writers about their urgent orders
     */
    private function notifyAssignedWriters(array $urgentOrders): void
    {
        $writerIds = collect($urgentOrders)
            ->flatten(1)
            ->pluck('writer_id')
            ->filter()
            ->unique();

        $writers = User::whereIn('id', $writerIds)->get();

        foreach ($writers as $writer) {
            try {
                // Get only orders assigned to this writer
                $writerOrders = collect($urgentOrders)
                    ->flatten(1)
                    ->where('writer_id', $writer->id)
                    ->toArray();

                if (!empty($writerOrders)) {
                    $writer->notify(new UrgentOrderNotification($writerOrders, 'writer'));
                }
            } catch (\Exception $e) {
                Log::error('Failed to send urgent order notification to writer', [
                    'writer_id' => $writer->id,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * Get urgent orders for a specific user (admin or writer)
     */
    public function getUrgentOrdersForUser(User $user): array
    {
        $urgentOrders = $this->getUrgentOrders();
        
        if ($user->user_type === 'writer') {
            // Filter to only show orders assigned to this writer
            $filteredOrders = collect($urgentOrders)
                ->map(function ($category) use ($user) {
                    return collect($category)
                        ->where('writer_id', $user->id)
                        ->toArray();
                })
                ->toArray();
            
            // Ensure all urgency levels are present, even if empty
            return [
                'overdue' => $filteredOrders['overdue'] ?? [],
                'critical_12h' => $filteredOrders['critical_12h'] ?? [],
                'urgent_24h' => $filteredOrders['urgent_24h'] ?? [],
                'warning_48h' => $filteredOrders['warning_48h'] ?? []
            ];
        }
        
        // Admin sees all urgent orders
        return $urgentOrders;
    }

    /**
     * Get urgent orders count for dashboard display
     */
    public function getUrgentOrdersCount(User $user): array
    {
        $urgentOrders = $this->getUrgentOrdersForUser($user);
        
        return [
            'overdue' => count($urgentOrders['overdue'] ?? []),
            'critical_12h' => count($urgentOrders['critical_12h'] ?? []),
            'urgent_24h' => count($urgentOrders['urgent_24h'] ?? []),
            'warning_48h' => count($urgentOrders['warning_48h'] ?? []),
            'total' => collect($urgentOrders)->flatten(1)->count()
        ];
    }
} 