<?php

namespace App\Services;

use App\Models\Order;
use App\Models\User;
use App\Notifications\NewOrderAvailableForBiddingNotification;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class WriterNotificationService
{
    /**
     * Notify all qualified writers about a new order available for bidding.
     *
     * @param Order $order
     * @return void
     */
    public function notifyQualifiedWriters(Order $order): void
    {
        try {
            // Get all qualified writers (those who have passed both tests)
            $qualifiedWriters = User::where('user_type', 'writer')
                ->whereHas('writerProfile', function ($query) {
                    $query->whereHas('qualificationTest', function ($q) {
                        $q->where('questions_test_status', 'passed');
                    })->whereHas('essayTest', function ($q) {
                        $q->where('essay_test_status', 'passed');
                    });
                })
                ->get();

            if ($qualifiedWriters->isEmpty()) {
                Log::info('No qualified writers found to notify about new order', [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number
                ]);
                return;
            }

            // Send notifications to all qualified writers
            Notification::send($qualifiedWriters, new NewOrderAvailableForBiddingNotification($order));

            Log::info('Successfully notified qualified writers about new order', [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'writers_notified' => $qualifiedWriters->count()
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to notify qualified writers about new order', [
                'error' => $e->getMessage(),
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Notify qualified writers about multiple orders available for bidding.
     *
     * @param \Illuminate\Support\Collection $orders
     * @return void
     */
    public function notifyQualifiedWritersBatch($orders): void
    {
        foreach ($orders as $order) {
            $this->notifyQualifiedWriters($order);
        }
    }

    /**
     * Get count of qualified writers for monitoring purposes.
     *
     * @return int
     */
    public function getQualifiedWritersCount(): int
    {
        return User::where('user_type', 'writer')
            ->whereHas('writerProfile', function ($query) {
                $query->whereHas('qualificationTest', function ($q) {
                    $q->where('questions_test_status', 'passed');
                })->whereHas('essayTest', function ($q) {
                    $q->where('essay_test_status', 'passed');
                });
            })
            ->count();
    }

    /**
     * Check if a specific writer is qualified.
     *
     * @param User $writer
     * @return bool
     */
    public function isWriterQualified(User $writer): bool
    {
        if ($writer->user_type !== 'writer') {
            return false;
        }

        $writerProfile = $writer->writerProfile;
        if (!$writerProfile) {
            return false;
        }

        return $writerProfile->isApproved();
    }
} 