<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderRevision;
use App\Models\OrderReassignment;
use App\Models\WriterFine;
use App\Models\WriterPaymentRecord;
use Carbon\Carbon;

class WriterPaymentCalculationService
{
    /**
     * Calculate the final payment amount for a writer considering all factors.
     */
    public function calculateFinalPayment(Order $order): array
    {
        $baseAmount = $order->writer_amount;
        $revisionAdjustments = $this->calculateRevisionAdjustments($order);
        $reassignmentAdjustments = $this->calculateReassignmentAdjustments($order);
        $fineDeductions = $this->calculateFineDeductions($order->writer_id);
        
        $finalAmount = $baseAmount + $revisionAdjustments['total'] + $reassignmentAdjustments['total'] - $fineDeductions;
        
        return [
            'base_amount' => $baseAmount,
            'revision_adjustments' => $revisionAdjustments,
            'reassignment_adjustments' => $reassignmentAdjustments,
            'fine_deductions' => $fineDeductions,
            'final_amount' => max(0, $finalAmount),
            'payment_status' => $this->determinePaymentStatus($order, $finalAmount),
        ];
    }

    /**
     * Calculate payment adjustments due to revisions.
     */
    private function calculateRevisionAdjustments(Order $order): array
    {
        $revisions = $order->revisions()->where('status', 'completed')->get();
        $totalPenalty = 0;
        $totalBonus = 0;
        $totalDelay = 0;
        $totalImpact = 0;
        
        foreach ($revisions as $revision) {
            // Use the calculated payment impact fields
            $totalPenalty += $revision->quality_penalty ?? 0;
            $totalBonus += $revision->quality_bonus ?? 0;
            $totalDelay += $revision->payment_delay_hours ?? 0;
            $totalImpact += $revision->payment_impact ?? 0;
        }
        
        return [
            'total' => $totalImpact, // Use the pre-calculated total impact
            'penalties' => $totalPenalty,
            'bonuses' => $totalBonus,
            'delay_hours' => $totalDelay,
            'revision_count' => $revisions->count(),
            'detailed_impacts' => $revisions->map(function ($revision) {
                return [
                    'revision_id' => $revision->id,
                    'type' => $revision->revision_type,
                    'quality_rating' => $revision->quality_rating,
                    'bonus' => $revision->quality_bonus,
                    'penalty' => $revision->quality_penalty,
                    'delay_penalty' => $revision->payment_delay_hours,
                    'total_impact' => $revision->payment_impact,
                ];
            }),
        ];
    }

    /**
     * Calculate payment adjustments due to reassignments.
     */
    private function calculateReassignmentAdjustments(Order $order): array
    {
        $reassignment = $order->latestReassignment;
        
        if (!$reassignment) {
            return [
                'total' => 0,
                'original_writer_payment' => 0,
                'new_writer_payment' => 0,
                'additional_cost' => 0,
            ];
        }
        
        // If this is the original writer, they get partial payment
        if ($reassignment->original_writer_id === $order->writer_id) {
            return [
                'total' => $reassignment->original_writer_payment,
                'original_writer_payment' => $reassignment->original_writer_payment,
                'new_writer_payment' => 0,
                'additional_cost' => 0,
            ];
        }
        
        // If this is the new writer, they get the remaining payment
        if ($reassignment->new_writer_id === $order->writer_id) {
            return [
                'total' => $reassignment->new_writer_payment,
                'original_writer_payment' => 0,
                'new_writer_payment' => $reassignment->new_writer_payment,
                'additional_cost' => 0,
            ];
        }
        
        return [
            'total' => 0,
            'original_writer_payment' => 0,
            'new_writer_payment' => 0,
            'additional_cost' => 0,
        ];
    }

    /**
     * Calculate total fine deductions for a writer.
     */
    private function calculateFineDeductions(int $writerId): float
    {
        return WriterFine::where('writer_id', $writerId)
            ->whereIn('status', ['pending', 'applied'])
            ->sum('remaining_amount');
    }

    /**
     * Determine the payment status based on various factors.
     */
    private function determinePaymentStatus(Order $order, float $finalAmount): string
    {
        if ($finalAmount <= 0) {
            return Order::WRITER_PAYMENT_STATUS_DISPUTED;
        }
        
        if ($order->hasActiveRevisions()) {
            return Order::WRITER_PAYMENT_STATUS_PENDING_REVISION;
        }
        
        if ($this->hasActiveFines($order->writer_id)) {
            return 'pending_with_fines';
        }
        
        if ($order->hasBeenReassigned()) {
            return 'partial_paid';
        }
        
        return Order::WRITER_PAYMENT_STATUS_AVAILABLE;
    }

    /**
     * Check if a writer has active fines.
     */
    private function hasActiveFines(int $writerId): bool
    {
        return WriterFine::where('writer_id', $writerId)
            ->whereIn('status', ['pending', 'applied'])
            ->exists();
    }

    /**
     * Process payment for an order, applying all adjustments and fines.
     */
    public function processPayment(Order $order): WriterPaymentRecord
    {
        $calculation = $this->calculateFinalPayment($order);
        
        // Create or update payment record
        $paymentRecord = $order->writerPaymentRecords()->firstOrCreate([
            'writer_id' => $order->writer_id,
        ]);
        
        $paymentRecord->update([
            'payment_amount' => $calculation['final_amount'],
            'status' => $this->mapPaymentStatus($calculation['payment_status']),
            'earned_at' => $order->order_status === 'approved' ? now() : null,
            'available_at' => $calculation['payment_status'] === 'available' ? now() : null,
        ]);
        
        // Apply fines if any
        $this->applyFinesToPayment($order->writer_id, $paymentRecord);
        
        // Update order payment status
        $order->update([
            'writer_payment_status' => $calculation['payment_status'],
            'writer_payment_calculated_at' => now(),
        ]);
        
        return $paymentRecord;
    }

    /**
     * Apply fines to a payment record.
     */
    private function applyFinesToPayment(int $writerId, WriterPaymentRecord $paymentRecord): void
    {
        $activeFines = WriterFine::where('writer_id', $writerId)
            ->whereIn('status', ['pending', 'applied'])
            ->get();
        
        foreach ($activeFines as $fine) {
            $fine->applyToPayment($paymentRecord);
        }
    }

    /**
     * Map internal payment status to WriterPaymentRecord status.
     */
    private function mapPaymentStatus(string $internalStatus): string
    {
        $statusMap = [
            'available' => WriterPaymentRecord::STATUS_AVAILABLE,
            'pending_revision' => WriterPaymentRecord::STATUS_PENDING,
            'pending_with_fines' => WriterPaymentRecord::STATUS_PENDING,
            'partial_paid' => WriterPaymentRecord::STATUS_PENDING,
            'disputed' => WriterPaymentRecord::STATUS_DISPUTED,
        ];
        
        return $statusMap[$internalStatus] ?? WriterPaymentRecord::STATUS_PENDING;
    }

    /**
     * Calculate payment delay due to revisions.
     */
    public function calculatePaymentDelay(Order $order): int
    {
        $revisions = $order->revisions()->where('status', 'completed')->get();
        $totalDelay = 0;
        
        foreach ($revisions as $revision) {
            $totalDelay += $revision->payment_delay_hours ?? 0;
        }
        
        return $totalDelay;
    }

    /**
     * Get payment summary for a writer across all orders.
     */
    public function getWriterPaymentSummary(int $writerId): array
    {
        $pendingOrders = Order::where('writer_id', $writerId)
            ->whereIn('writer_payment_status', ['pending', 'pending_revision'])
            ->get();
        
        $availableOrders = Order::where('writer_id', $writerId)
            ->where('writer_payment_status', 'available')
            ->get();
        
        $withdrawnOrders = Order::where('writer_id', $writerId)
            ->where('writer_payment_status', 'withdrawn')
            ->get();
        
        $totalPending = $this->calculateTotalAmount($pendingOrders);
        $totalAvailable = $this->calculateTotalAmount($availableOrders);
        $totalWithdrawn = $this->calculateTotalAmount($withdrawnOrders);
        
        return [
            'pending_amount' => $totalPending,
            'available_amount' => $totalAvailable,
            'withdrawn_amount' => $totalWithdrawn,
            'total_earned' => $totalPending + $totalAvailable + $totalWithdrawn,
            'pending_orders_count' => $pendingOrders->count(),
            'available_orders_count' => $availableOrders->count(),
            'withdrawn_orders_count' => $withdrawnOrders->count(),
        ];
    }

    /**
     * Calculate total amount for a collection of orders.
     */
    private function calculateTotalAmount($orders): float
    {
        $total = 0;
        
        foreach ($orders as $order) {
            $calculation = $this->calculateFinalPayment($order);
            $total += $calculation['final_amount'];
        }
        
        return $total;
    }
} 