#!/bin/bash

# WebP Image Conversion Script
# Converts all JPG and PNG images to WebP format while preserving aspect ratios

echo "🚀 Starting WebP conversion process..."
echo ""

# Base directory
BASE_DIR="/home/beenie/Documents/academic-scribe/public/images"

# Counters
TOTAL=0
CONVERTED=0
SKIPPED=0
ERRORS=0

# Quality settings
QUALITY=85
LOGO_QUALITY=90

# Function to convert image
convert_image() {
    local input_file="$1"
    local output_file="${input_file%.*}.webp"
    local quality="${2:-$QUALITY}"
    
    # Skip if already a WebP file
    if [[ "$input_file" == *.webp ]]; then
        return
    fi
    
    # Skip if WebP already exists (unless force mode)
    if [ -f "$output_file" ] && [ "$FORCE" != "true" ]; then
        echo "⏭️  Skipping (exists): $(basename "$output_file")"
        ((SKIPPED++))
        return
    fi
    
    # Get file size before
    if [ -f "$input_file" ]; then
        SIZE_BEFORE=$(stat -f%z "$input_file" 2>/dev/null || stat -c%s "$input_file" 2>/dev/null)
    else
        echo "❌ File not found: $input_file"
        ((ERRORS++))
        return
    fi
    
    # Convert to WebP
    if cwebp -q "$quality" "$input_file" -o "$output_file" > /dev/null 2>&1; then
        SIZE_AFTER=$(stat -f%z "$output_file" 2>/dev/null || stat -c%s "$output_file" 2>/dev/null)
        SAVED=$((SIZE_BEFORE - SIZE_AFTER))
        
        echo "✅ Converted: $(basename "$input_file")"
        echo "   📊 Before: $(numfmt --to=iec-i --suffix=B $SIZE_BEFORE 2>/dev/null || echo "${SIZE_BEFORE} bytes")"
        echo "   📊 After: $(numfmt --to=iec-i --suffix=B $SIZE_AFTER 2>/dev/null || echo "${SIZE_AFTER} bytes")"
        if [ $SAVED -gt 0 ]; then
            echo "   💾 Saved: $(numfmt --to=iec-i --suffix=B $SAVED 2>/dev/null || echo "${SAVED} bytes")"
        fi
        ((CONVERTED++))
    else
        echo "❌ Failed: $(basename "$input_file")"
        ((ERRORS++))
    fi
}

# Check if force mode
FORCE="${1:-false}"
if [ "$1" == "--force" ] || [ "$1" == "-f" ]; then
    FORCE="true"
    echo "⚠️  FORCE MODE: Will overwrite existing WebP files"
    echo ""
fi

# Convert logo with higher quality
echo "📁 Converting Logo..."
if [ -f "$BASE_DIR/logo.png" ]; then
    convert_image "$BASE_DIR/logo.png" $LOGO_QUALITY
    ((TOTAL++))
fi
echo ""

# Convert hero images
echo "📁 Converting Hero Images..."
for img in "$BASE_DIR"/hero-*.jpg "$BASE_DIR"/hero-*.png; do
    if [ -f "$img" ]; then
        convert_image "$img"
        ((TOTAL++))
    fi
done
echo ""

# Convert main directory images
echo "📁 Converting Main Images..."
for ext in jpg jpeg png; do
    for img in "$BASE_DIR"/*.$ext; do
        if [ -f "$img" ]; then
            convert_image "$img"
            ((TOTAL++))
        fi
    done
done
echo ""

# Convert blog images
echo "📁 Converting Blog Images..."
for ext in jpg jpeg png; do
    for img in "$BASE_DIR/blog"/*.$ext; do
        if [ -f "$img" ]; then
            convert_image "$img"
            ((TOTAL++))
        fi
    done
done
echo ""

# Convert author images
echo "📁 Converting Author Images..."
for ext in jpg jpeg png; do
    for img in "$BASE_DIR/authors"/*.$ext; do
        if [ -f "$img" ]; then
            convert_image "$img"
            ((TOTAL++))
        fi
    done
done
echo ""

# Convert service images
echo "📁 Converting Service Images..."
for ext in jpg jpeg png; do
    for img in "$BASE_DIR/services"/*.$ext; do
        if [ -f "$img" ]; then
            convert_image "$img"
            ((TOTAL++))
        fi
    done
done
echo ""

# Display results
echo "================================================"
echo "📊 CONVERSION RESULTS"
echo "================================================"
echo "📈 Total images found: $TOTAL"
echo "✅ Successfully converted: $CONVERTED"
echo "⏭️  Skipped (already exist): $SKIPPED"
echo "❌ Errors: $ERRORS"
echo ""

if [ $CONVERTED -gt 0 ]; then
    echo "🎉 WebP conversion complete!"
else
    echo "⚠️  No images were converted."
    echo "💡 Use --force to overwrite existing WebP files"
fi

echo ""
echo "Usage: ./convert-images-to-webp.sh [--force]"




