<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateOrderRevisionsTable extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('order_revisions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('order_id')->constrained()->onDelete('cascade');
            $table->foreignId('requested_by')->constrained('users')->comment('Admin who requested revision');
            $table->foreignId('assigned_to')->nullable()->constrained('users')->comment('Writer assigned to revision');
            
            // Revision details
            $table->enum('revision_type', [
                'quality', 
                'requirements', 
                'formatting', 
                'plagiarism', 
                'deadline', 
                'communication', 
                'other'
            ]);
            $table->text('revision_reason');
            $table->text('specific_instructions')->nullable();
            
            // Revision status tracking
            $table->enum('status', [
                'requested', 
                'in_progress', 
                'completed', 
                'rejected', 
                'escalated'
            ])->default('requested');
            
            // Timeline tracking
            $table->timestamp('requested_at');
            $table->timestamp('due_date')->nullable();
            $table->timestamp('started_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->timestamp('rejected_at')->nullable();
            
            // Quality assessment
            $table->integer('quality_rating')->nullable()->comment('1-5 rating of revision quality');
            $table->text('quality_notes')->nullable();
            
            // Payment impact
            $table->decimal('payment_delay_hours', 8, 2)->default(0)->comment('Hours payment is delayed');
            $table->decimal('quality_penalty', 8, 2)->default(0)->comment('Monetary penalty for poor quality');
            $table->decimal('quality_bonus', 8, 2)->default(0)->comment('Monetary bonus for excellent quality');
            
            // Revision limits
            $table->integer('revision_number')->default(1)->comment('Which revision this is (1st, 2nd, etc.)');
            $table->boolean('is_final_revision')->default(false)->comment('If this is the final allowed revision');
            
            // Communication tracking
            $table->foreignId('conversation_id')->nullable()->constrained()->comment('Linked conversation thread');
            
            // Admin notes
            $table->text('admin_notes')->nullable();
            $table->text('escalation_reason')->nullable();
            
            $table->timestamps();
            
            // Indexes for performance
            $table->index(['order_id', 'status']);
            $table->index(['assigned_to', 'status']);
            $table->index('due_date');
            $table->index('revision_number');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('order_revisions');
    }
} 