<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateOrderReassignmentsTable extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('order_reassignments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('order_id')->constrained()->onDelete('cascade');
            $table->foreignId('original_writer_id')->constrained('users')->comment('Original writer assigned to order');
            $table->foreignId('new_writer_id')->constrained('users')->comment('New writer assigned to order');
            $table->foreignId('reassigned_by')->constrained('users')->comment('Admin who made the reassignment');
            
            // Reassignment details
            $table->enum('reassignment_reason', [
                'quality_issues',
                'deadline_violation',
                'communication_failure',
                'writer_unavailable',
                'client_request',
                'workload_balance',
                'expertise_required',
                'other'
            ]);
            $table->text('detailed_reason');
            $table->text('admin_notes')->nullable();
            
            // Order progress at reassignment
            $table->decimal('progress_percentage', 5, 2)->default(0)->comment('Percentage of work completed');
            $table->enum('order_status_at_reassignment', [
                'assigned',
                'in_progress',
                'submitted',
                'under_review',
                'revision_requested'
            ]);
            
            // Timeline tracking
            $table->timestamp('reassigned_at');
            $table->timestamp('work_handover_completed_at')->nullable();
            $table->timestamp('new_writer_started_at')->nullable();
            
            // Payment calculations
            $table->decimal('original_writer_payment', 10, 2)->default(0)->comment('Payment for original writer');
            $table->decimal('new_writer_payment', 10, 2)->default(0)->comment('Payment for new writer');
            $table->decimal('total_payment', 10, 2)->default(0)->comment('Total payment across both writers');
            
            // Work handover
            $table->text('work_handover_notes')->nullable();
            $table->json('files_transferred')->nullable()->comment('List of files transferred');
            $table->text('requirements_summary')->nullable()->comment('Summary of work requirements');
            
            // Quality assessment
            $table->integer('original_work_quality_rating')->nullable()->comment('1-5 rating of original work');
            $table->text('quality_assessment_notes')->nullable();
            
            // Reassignment impact
            $table->decimal('delay_hours', 8, 2)->default(0)->comment('Hours delayed due to reassignment');
            $table->decimal('additional_cost', 10, 2)->default(0)->comment('Additional cost due to reassignment');
            
            $table->timestamps();
            
            // Indexes for performance
            $table->index(['order_id']);
            $table->index(['original_writer_id', 'reassigned_at']);
            $table->index(['new_writer_id', 'reassigned_at']);
            $table->index('reassigned_at');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('order_reassignments');
    }
} 