<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

return new class extends Migration
{
    public function up(): void
    {
        DB::transaction(function () {
            $pages = [
                'affordable-essay-writing' => 'affordable',
                'buy-essays-online' => 'buy',
                'cheap-essays-online' => 'cheap',
            ];

            foreach ($pages as $slug => $prefix) {
                $pageId = DB::table('pages')->where('slug', $slug)->value('id');

                if (!$pageId) {
                    continue;
                }

                $coreMap = [
                    'hero' => 'service_hero',
                    'overview' => 'service_overview',
                    'process' => 'service_process',
                    'benefits' => 'service_benefits',
                    'testimonials' => 'service_testimonials',
                    'faqs' => 'service_faqs',
                    'cta' => 'service_cta',
                ];

                foreach ($coreMap as $suffix => $blockType) {
                    $blockKey = $prefix . '_' . $suffix;

                    DB::table('page_content')
                        ->where('page_id', $pageId)
                        ->where('block_key', $blockKey)
                        ->update(['block_type' => $blockType]);
                }

                $linkMap = [
                    'related_services' => 'related',
                    'support_pages' => 'support',
                    'resource_links' => 'resources',
                ];

                foreach ($linkMap as $suffix => $category) {
                    $blockKey = $prefix . '_' . $suffix;
                    $row = DB::table('page_content')
                        ->where('page_id', $pageId)
                        ->where('block_key', $blockKey)
                        ->first();

                    if (!$row) {
                        continue;
                    }

                    $content = json_decode($row->content, true) ?? [];
                    $content['linkCategory'] = $content['linkCategory'] ?? $category;

                    if (!isset($content['description_html']) && isset($content['description'])) {
                        $content['description_html'] = $content['description'];
                    }

                    DB::table('page_content')
                        ->where('id', $row->id)
                        ->update([
                            'block_type' => 'service_links',
                            'content' => json_encode($content),
                        ]);
                }

                $longformSuffixes = [
                    'trust',
                    'writer_selection',
                    'reasons',
                    'originality',
                    'join_experts',
                    'types',
                    'hiring_steps',
                    'writer_standout',
                    'service_benefits',
                    'peace_of_mind',
                    'quality_score',
                ];

                foreach ($longformSuffixes as $suffix) {
                    $legacyKey = $prefix . '_longform_' . $suffix;
                    $originalKey = $prefix . '_' . $suffix;

                    $row = DB::table('page_content')
                        ->where('page_id', $pageId)
                        ->where('block_key', $legacyKey)
                        ->first();

                    if (!$row) {
                        $row = DB::table('page_content')
                            ->where('page_id', $pageId)
                            ->where('block_key', $originalKey)
                            ->first();
                    }

                    if (!$row) {
                        continue;
                    }

                    $content = json_decode($row->content, true) ?? [];
                    $sections = $content['sections'] ?? [];
                    $html = $content['description_html'] ?? $content['description'] ?? null;

                    if (!$html && !empty($sections)) {
                        $html = $this->combineSectionsHtml($sections);
                    }

                    $content['eyebrow'] = $content['eyebrow'] ?? 'Deep Dive';
                    $content['title'] = $content['title'] ?? Str::headline(str_replace('_', ' ', $suffix));
                    $content['description'] = $html;
                    $content['description_html'] = $html;
                    $content['preferredHeight'] = isset($content['preferredHeight'])
                        ? (int) $content['preferredHeight']
                        : 480;
                    $content['sections'] = [];

                    DB::table('page_content')
                        ->where('id', $row->id)
                        ->update([
                            'block_key' => $originalKey,
                            'block_type' => 'service_scrollable',
                            'content' => json_encode($content),
                        ]);
                }
            }
        });
    }

    public function down(): void
    {
        DB::transaction(function () {
            $pages = [
                'affordable-essay-writing' => 'affordable',
                'buy-essays-online' => 'buy',
                'cheap-essays-online' => 'cheap',
            ];

            foreach ($pages as $slug => $prefix) {
                $pageId = DB::table('pages')->where('slug', $slug)->value('id');

                if (!$pageId) {
                    continue;
                }

                $coreMap = [
                    'hero' => 'hero',
                    'overview' => 'overview',
                    'process' => 'process',
                    'benefits' => 'benefits',
                    'testimonials' => 'testimonials',
                    'faqs' => 'faqs',
                    'cta' => 'cta',
                ];

                foreach ($coreMap as $suffix => $blockType) {
                    $blockKey = $prefix . '_' . $suffix;

                    DB::table('page_content')
                        ->where('page_id', $pageId)
                        ->where('block_key', $blockKey)
                        ->update(['block_type' => $blockType]);
                }

                $linkSuffixes = ['related_services', 'support_pages', 'resource_links'];

                foreach ($linkSuffixes as $suffix) {
                    $blockKey = $prefix . '_' . $suffix;

                    DB::table('page_content')
                        ->where('page_id', $pageId)
                        ->where('block_key', $blockKey)
                        ->update(['block_type' => 'links']);
                }

                $longformSuffixes = [
                    'trust',
                    'writer_selection',
                    'reasons',
                    'originality',
                    'join_experts',
                    'types',
                    'hiring_steps',
                    'writer_standout',
                    'service_benefits',
                    'peace_of_mind',
                    'quality_score',
                ];

                foreach ($longformSuffixes as $suffix) {
                    $legacyKey = $prefix . '_longform_' . $suffix;
                    $originalKey = $prefix . '_' . $suffix;

                    $row = DB::table('page_content')
                        ->where('page_id', $pageId)
                        ->where('block_key', $originalKey)
                        ->first();

                    if (!$row) {
                        continue;
                    }

                    $content = json_decode($row->content, true) ?? [];
                    $html = $content['description_html'] ?? $content['description'] ?? '';

                    $content['sections'] = [
                        [
                            'title' => null,
                            'body_html' => $html,
                            'body' => $html,
                            'is_active' => true,
                        ],
                    ];

                    DB::table('page_content')
                        ->where('id', $row->id)
                        ->update([
                            'block_key' => $legacyKey,
                            'block_type' => 'service_longform',
                            'content' => json_encode($content),
                        ]);
                }
            }
        });
    }

    private function combineSectionsHtml(array $sections): string
    {
        return collect($sections)
            ->filter(fn ($section) => isset($section['body']) || isset($section['body_html']))
            ->filter(fn ($section) => $section['is_active'] ?? true)
            ->map(function ($section) {
                $title = trim((string) ($section['title'] ?? ''));
                $body = $section['body_html'] ?? $section['body'] ?? '';

                if ($body === '') {
                    return '';
                }

                $heading = $title !== '' ? '<h3>' . e($title) . '</h3>' : '';

                return $heading . $body;
            })
            ->filter()
            ->implode('<br /><br />');
    }
};
