<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

return new class extends Migration
{
    private array $prefixToSlug = [
        'affordable' => 'affordable-essay-writing',
        'buy' => 'buy-essays-online',
        'cheap' => 'cheap-essays-online',
    ];

    public function up(): void
    {
        DB::transaction(function () {
            foreach ($this->prefixToSlug as $prefix => $slug) {
                $blockKey = "{$prefix}_faqs";

                $row = DB::table('page_content')
                    ->where('block_key', $blockKey)
                    ->first();

                if (!$row) {
                    continue;
                }

                $content = json_decode($row->content, true) ?? [];

                if (!empty($content['faqs'])) {
                    continue; // already structured
                }

                $html = $content['description_html'] ?? $content['description'] ?? '';

                $faqs = [];

                if ($html) {
                    $faqs = $this->parseFaqsFromHtml($html);
                }

                if (empty($faqs)) {
                    $markdownPath = resource_path("content/services/{$slug}/{$prefix}_faqs.md");
                    if (File::exists($markdownPath)) {
                        $markdown = File::get($markdownPath);
                        $faqs = $this->parseFaqsFromMarkdown($markdown);
                    }
                }

                if (empty($faqs)) {
                    continue;
                }

                $content['faqs'] = $faqs;
                $content['description'] = '<p>Find answers to the most common questions about this service.</p>';
                $content['description_html'] = $content['description'];

                DB::table('page_content')
                    ->where('id', $row->id)
                    ->update([
                        'content' => json_encode($content),
                    ]);
            }
        });
    }

    public function down(): void
    {
        // Left intentionally blank. Structured FAQ data should persist.
    }

    private function parseFaqsFromHtml(string $html): array
    {
        $faqs = [];
        $cursor = 0;
        $length = mb_strlen($html);

        while (true) {
            $qPos = $this->stripos($html, 'Q:', $cursor);

            if ($qPos === false) {
                break;
            }

            $aPos = $this->stripos($html, 'A:', $qPos);

            if ($aPos === false) {
                break;
            }

            $nextQ = $this->stripos($html, 'Q:', $aPos + 2);

            $questionChunk = mb_substr($html, $qPos + 2, $aPos - ($qPos + 2));
            $answerChunk = $nextQ === false
                ? mb_substr($html, $aPos + 2)
                : mb_substr($html, $aPos + 2, $nextQ - ($aPos + 2));

            $question = trim(strip_tags($questionChunk));
            $answerHtml = trim($answerChunk);

            $answerHtml = $this->stripLeadingHeadings($answerHtml);

            if ($question === '' || $answerHtml === '') {
                $cursor = $aPos + 2;
                continue;
            }

            $faqs[] = [
                'question' => $question,
                'answer' => trim(strip_tags($answerHtml)),
                'answer_html' => $answerHtml,
                'is_active' => true,
            ];

            $cursor = $nextQ === false ? $length : $nextQ;
        }

        return $faqs;
    }

    private function parseFaqsFromMarkdown(string $markdown): array
    {
        $faqs = [];
        $lines = preg_split('/\r?\n/', $markdown);
        $currentQuestion = null;
        $currentAnswer = [];

        foreach ($lines as $line) {
            $trimmed = trim($line);

            if (preg_match('/^\*\*(.+?)\*\*$/', $trimmed, $matches)) {
                if ($currentQuestion !== null) {
                    $faqs[] = $this->buildFaq($currentQuestion, $currentAnswer);
                    $currentAnswer = [];
                }

                $currentQuestion = trim($matches[1]);
            } elseif ($currentQuestion !== null) {
                if (str_starts_with($trimmed, '##')) {
                    continue;
                }
                $currentAnswer[] = $line;
            }
        }

        if ($currentQuestion !== null) {
            $faqs[] = $this->buildFaq($currentQuestion, $currentAnswer);
        }

        return array_values(array_filter($faqs));
    }

    private function buildFaq(?string $question, array $answerLines): ?array
    {
        if (!$question) {
            return null;
        }

        $answerText = trim(implode("\n", $answerLines));

        if ($answerText === '') {
            return null;
        }

        $answerHtml = Str::markdown($answerText);
        $answerHtml = $this->stripLeadingHeadings($answerHtml);

        if ($answerHtml === '') {
            return null;
        }

        return [
            'question' => $question,
            'answer' => trim(strip_tags($answerHtml)),
            'answer_html' => $answerHtml,
            'is_active' => true,
        ];
    }

    private function stripos(string $haystack, string $needle, int $offset = 0)
    {
        $pos = mb_stripos($haystack, $needle, $offset, 'UTF-8');

        return $pos === false ? false : $pos;
    }

    private function stripLeadingHeadings(string $html): string
    {
        $result = preg_replace('/^(<h[1-6][^>]*>.*?<\/h[1-6]>\s*)+/is', '', $html);

        return trim($result ?? '');
    }
};
