<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

return new class extends Migration
{
    private array $prefixes = ['affordable', 'buy', 'cheap'];
    private array $prefixToSlug = [
        'affordable' => 'affordable-essay-writing',
        'buy' => 'buy-essays-online',
        'cheap' => 'cheap-essays-online',
    ];

    public function up(): void
    {
        DB::transaction(function () {
            foreach ($this->prefixes as $prefix) {
                $blockKey = "{$prefix}_faqs";

                $row = DB::table('page_content')
                    ->where('block_key', $blockKey)
                    ->first();

                if (!$row) {
                    continue;
                }

                $content = json_decode($row->content, true) ?? [];

                if (empty($content['faqs']) || !is_array($content['faqs'])) {
                    $markdownPath = resource_path("content/services/{$this->prefixToSlug[$prefix]}/{$prefix}_faqs.md");
                    if (File::exists($markdownPath)) {
                        $markdown = File::get($markdownPath);
                        $content['faqs'] = $this->parseFaqsFromMarkdown($markdown);
                    }
                }

                if (empty($content['faqs']) || !is_array($content['faqs'])) {
                    continue;
                }

                $updatedFaqs = [];

                foreach ($content['faqs'] as $faq) {
                    $answerHtml = $faq['answer_html'] ?? $faq['answer'] ?? '';
                    $answerHtml = $this->stripLeadingHeadings($answerHtml);

                    if ($answerHtml === '') {
                        continue;
                    }

                    $updatedFaqs[] = [
                        'question' => $faq['question'] ?? '',
                        'answer' => trim(strip_tags($answerHtml)),
                        'answer_html' => $answerHtml,
                        'is_active' => $faq['is_active'] ?? true,
                    ];
                }

                if (empty($updatedFaqs)) {
                    continue;
                }

                $content['faqs'] = $updatedFaqs;
                $content['description'] = $content['description'] ?? '<p>Find answers to the most common questions about this service.</p>';
                $content['description_html'] = $content['description_html'] ?? $content['description'];

                DB::table('page_content')
                    ->where('id', $row->id)
                    ->update([
                        'content' => json_encode($content),
                    ]);
            }
        });
    }

    public function down(): void
    {
        // No rollback required.
    }

    private function stripLeadingHeadings(string $html): string
    {
        $html = trim($html);
        if ($html === '') {
            return '';
        }

        $clean = preg_replace('/^(<h[1-6][^>]*>.*?<\/h[1-6]>\s*)+/is', '', $html);

        return trim($clean ?? '');
    }

    private function parseFaqsFromMarkdown(string $markdown): array
    {
        $faqs = [];
        $lines = preg_split('/\r?\n/', $markdown);
        $currentQuestion = null;
        $currentAnswer = [];

        foreach ($lines as $line) {
            $trimmed = trim($line);

            if (preg_match('/^\*\*(.+?)\*\*$/', $trimmed, $matches)) {
                if ($currentQuestion !== null) {
                    $faq = $this->buildFaq($currentQuestion, $currentAnswer);
                    if ($faq) {
                        $faqs[] = $faq;
                    }
                    $currentAnswer = [];
                }

                $currentQuestion = trim($matches[1]);
            } elseif ($currentQuestion !== null) {
                if (str_starts_with($trimmed, '##')) {
                    continue;
                }
                $currentAnswer[] = $line;
            }
        }

        if ($currentQuestion !== null) {
            $faq = $this->buildFaq($currentQuestion, $currentAnswer);
            if ($faq) {
                $faqs[] = $faq;
            }
        }

        return array_values(array_filter($faqs));
    }

    private function buildFaq(?string $question, array $answerLines): ?array
    {
        if (!$question) {
            return null;
        }

        $answerText = trim(implode("\n", $answerLines));

        if ($answerText === '') {
            return null;
        }

        $answerHtml = Str::markdown($answerText);
        $answerHtml = $this->stripLeadingHeadings($answerHtml);

        if ($answerHtml === '') {
            return null;
        }

        return [
            'question' => $question,
            'answer' => trim(strip_tags($answerHtml)),
            'answer_html' => $answerHtml,
            'is_active' => true,
        ];
    }
};
